/* MAX17043.cpp
 * Tested with mbed board: FRDM-KL46Z
 * Author: Mark Gottscho
 * mgottscho@ucla.edu
 */


#include "mbed.h"
#include "MAX17043.h"
#include <new>

MAX17043::MAX17043(PinName sda, PinName scl, int i2c_addr) : i2c_(*reinterpret_cast<I2C*>(i2cRaw)), __i2c_addr(i2c_addr)
{
    // Placement new to avoid additional heap memory allocation.
    new(i2cRaw) I2C(sda, scl);

    selfInit();
}

MAX17043::MAX17043(I2C &i2c, int i2c_addr): i2c_(i2c), __i2c_addr(i2c_addr)
{
    selfInit();
}

MAX17043::~MAX17043() { }

void MAX17043::selfInit()
{
    __soc = 0;
    __vcell = 0;
    reset();
}

void MAX17043::reset()
{
    uint16_t data = RST_CODE;
    setRegister16b(COMMAND_MSB, data);
    wait(0.130); //wait 130ms until first readings are valid (125ms est)
}

uint16_t MAX17043::getVersion()
{
    uint8_t data = getRegister(VERSION_MSB);
    return (data << 8) | (getRegister(VERSION_LSB));
}

uint16_t MAX17043::getVCell()
{
    uint16_t data = getRegister16b(VCELL_MSB);
    __vcell = data >> 4; //right shift by 4 to throw out the don't care bits
    return __vcell;
}

float MAX17043::getFloatVCell()
{
    return getVCell() * DIV_VCELL;
}

uint16_t MAX17043::getSOC()
{
    uint8_t data = getRegister(SOC_MSB);
    __soc =  (data << 8) | (getRegister(SOC_LSB));
    return __soc;
}

float MAX17043::getFloatSOC()
{
    return getSOC() * DIV_SOC;
}


void MAX17043::setRegister16b(const uint8_t reg_addr, const uint16_t data)
{
    uint8_t dataMSB = (data >> 8) & 0x00FF;
    uint8_t dataLSB = data & 0x00FF;
    uint8_t payload[3] = {reg_addr, dataMSB, dataLSB};
    __writeReg(payload, 3);
}
void MAX17043::setRegister(const uint8_t reg_addr, const uint8_t data)
{
    uint8_t payload[2] = {reg_addr, data};
    __writeReg(payload, 2);
}

uint8_t MAX17043::getRegister(const uint8_t reg_addr)
{
    uint8_t data;
    __readReg(reg_addr, &data, 1);
    return data;
}

uint16_t MAX17043::getRegister16b(const uint8_t reg_addr)
{
    uint8_t payload[2];
    __readReg(reg_addr, payload, 2);
    uint16_t data = (payload[0] << 8) | (payload[1]);
    return data;
}
int MAX17043::__readReg(const uint8_t reg_addr, uint8_t *data, int len)
{
    int retval = 0;

    retval = i2c_.write(__i2c_addr, (char *) &reg_addr, 1, true);
    if (retval != 0) {
        return retval;
    }
    retval = i2c_.read(__i2c_addr, (char *) data, len);

    return retval;
}

int MAX17043::__writeReg(const uint8_t *data, int total_len)
{
    int retval = 0;
    retval = i2c_.write(__i2c_addr, (char *) data, total_len);

    return retval;
}
