// This example program will make a cellular connection, open a TCP socket then
// send and receive data on the socket. After transmission, it will close
// the socket and disconnect the cellular connection.
//
// ** Configure the apn[] value below for MTQ-H5 or MTQ-LAT3 radios.
// ** INTERVAL sets the number of seconds between repeated cycles of this example.
 
#include "mbed.h"
#include "MTSCellularInterface.h"
#include "MTSLog.h"
#include "example_config.h"
 
#if ACTIVE_EXAMPLE == TCP_EXAMPLE
 
// Dragonfly debug port. 
Serial debug_port(USBTX, USBRX);
 
#define INTERVAL 30
 
int main(){
    //Sets the log level to INFO, higher log levels produce more log output.
    //Possible levels: NONE, FATAL, ERROR, WARNING, INFO, DEBUG and TRACE.
    //For the Dragonfly, installed on the UDK 2.0 board, messages are output on the
    // UDK 2.0 USB port “COMxx: STMicroelectronics STLink Virtual COM port (xx)”
    // at 9600bps.    
    mts::MTSLog::setLogLevel(mts::MTSLog::INFO_LEVEL);
    //Sets the debug port to 115200bps.
    debug_port.baud(9600);
 
    // Create an MTSCellularInterface object. Serial pins for the Dragonfly board that connect
    // to the on board cellular radio:
    // RADIO_TX = pin PC_7, RADIO_RX = pin PC_6
    MTSCellularInterface *radio = new MTSCellularInterface(RADIO_TX, RADIO_RX);
 
    // Print the MTSCellularInterface version
    logInfo("MTSCellularInterface Version %s", radio->get_library_version().c_str());
 
    //Modify to match your apn if you are using the MTQ-H5 or MTQ-LAT3.
    const char apn[] = "rogers-core-appl1.apn";
 
    // Basic HTTP request.
    std::string request = "POST / HTTP/1.1\r\nHost: httpbin.org/post\r\n\r\n";
 
    int cycle_count = 1;
    
    while (true) {       
        logInfo("-------- CYCLE #%d --------\r\n", cycle_count++);
        
        while(!radio->is_registered()){
            logWarning("radio not registered, try again in 2s");
            wait(2);
        }
 
        Timer tmr;  //mbed Timer has a 30 minute maximum timeout.
        tmr.start();
        while (radio->connect(apn) != NSAPI_ERROR_OK) {
            logWarning("Radio did not connect");
            while (tmr.read() < 30);    //make sure we wait at least 30s.
            tmr.reset();
        }
        tmr.stop();
 
        // Show the network address
        const char *ip = radio->get_ip_address();
        logInfo("IP address is: %s\n", ip ? ip : "No IP");
 
        // Open a socket on the network interface, and create a TCP connection to mbed.org
        
        TCPSocket socket;
 
        // Open a socket on the network interface.
        if (socket.open(radio) != NSAPI_ERROR_OK) {
            logWarning("socket did not open");
            socket.close();
            continue;
        }
 
        // Make a socket connection.
        if (socket.connect("httpbin.org", 80) != NSAPI_ERROR_OK) {
            logWarning("socket did not connect");
            socket.close();
            continue;
        }
 
        // Send tcp data
        int scount = socket.send(request.c_str(), request.size());
        logInfo("sent %d bytes: %s", scount, request.c_str());
 
        // Recieve and print. Give a couple seonds to receive.
        int size = 512;
        char rbuffer[size];
        memset(rbuffer, 0, size);
        bool got_data = false;
        Timer rcv_timer;
        rcv_timer.start();
        do {
            int rcount = socket.recv(rbuffer, size-1);  //leave room for a null character
            if (rcount > 0) {
                got_data = true;
                while (rcount > 0) {                    
                    logInfo("recv %d bytes: %s", rcount, rbuffer);
                    memset(rbuffer, 0, size);
                    rcount = socket.recv(rbuffer, size-1);
                }
            }
        } while (rcv_timer < 2 && !got_data);
        
        
        //char rbuffer[64];
        //int rcount = socket.recv(rbuffer, sizeof rbuffer);
        //printf("recv %d [%.*s]\n", rcount, strstr(rbuffer, "\r\n")-rbuffer, rbuffer);
        
        
        // Close the socket to return its memory and bring down the network interface
        socket.close();
 
        radio->disconnect();
 
        logInfo("waiting %d seconds\r\n", INTERVAL);
        wait(INTERVAL);
    }
        
}
 
#endif
 
         