/*  Pneumoscope Version 1.0
    Florian CHAYS
 */

#include "mbed.h"
#include <main.h>

// ==== Main ====
int main() {
    printf("\n\r==== AFE TestBench ====\n\n\r");

    AFE_begin();
    // Send Defaut Register Values
    for (int i = 0; i< 49; i++){
        SPI_Write(i,AFE_Default[i]);
    }
    AFE_set_LED_current(255,255);
    
    // Print Register State
    printf("== Default Register State ==\n\r");
    AFE_register_state();
    
    // Register State Control
    AFE_control_register();
    
    // Mesurements
    AFE_measurement();
    
    // Diagnostic Flags
    AFE_diag_flag();
    
    // Results
    AFE_Results();
}



// ===================
// ==== Functions ====
// ===================

void AFE_reset(){
    SPI_Write_Bit(CONTROL0, 3, true);
    while((SPI_Read(CONTROL0) >> 3) & 0x1); // Waiting for Reset bit back to 0
}

void AFE_begin(){
    // Supply Enable
    SUP_EN.write(1);
    
    // Power Up
    AFE_PDN.write(1);
    
    // Hard Reset
    AFE_nRST.write(0);
    ThisThread::sleep_for(500);
    AFE_nRST.write(1);
    myled.write(1);
    
    // Soft Reset
    AFE_reset();
}

void AFE_set_LED_current(uint8_t led1_current, uint8_t led2_current){
    // read the reg
    uint32_t current_val = SPI_Read(LEDCNTRL);
    
    // set led 1 value
    led1_current = led1_current & 0xFF;
    current_val = current_val & ~(0xFF << 8);
    current_val = current_val | (led1_current << 8);
    
    // set led 2 value
    led2_current = led2_current & 0xFF;
    current_val = current_val & ~(0xFF);
    current_val = current_val | (led2_current);

    // write reg
    SPI_Write(LEDCNTRL, current_val);
}

void AFE_measurement(){
    printf("== Start Measures ==\n\r");
    SPI_Write_Bit(CONTROL2, 11, false); //Set default led state
    SPI_Write_Bit(LEDCNTRL, 17, true);
    SPI_Write_Bit(CONTROL1, 8, true);
    printf("Done\n\r");
    
    printf("== Diagnostic ==\n\r");
    SPI_Write_Bit(CONTROL0, 2, true);   // Start Diagnosis
    ThisThread::sleep_for(500);
    //while(!AFE_DIAG_END);               // Waiting for Diag_En bit going high
    printf("Done\n\r");
}

void AFE_register_state(){
    uint8_t address = 0;
    uint32_t data_out = 0;
    for (int i = 0 ; address <  34 ; i++ ) {
        printf("\n\r%d : ", address ) ;
        for (int j = 0 ; j < 8 ; j++ ) {
            data_out = SPI_Read(address) ;
            printf("%lu ", data_out) ;
            //printf("%x ", data_out);
            address++;
        }
    }
    printf("\n\n\r");
}

void AFE_control_register(){
    uint32_t data_out = 0;
    bool nr_error = 0;
    for (char address = 1; address < 34; address++){
        data_out = SPI_Read(address);
        nr_error = error_check(address, AFE_Default[address], data_out);
    }
}

void AFE_diag_flag(){
    uint16_t flags = 0;
    flags = SPI_Read(0x30) & 0x1FFF;
    if (flags){
        for (int i = 0; i < 13; i++){
            if ((flags >> i) & 0x1){
                printf("[ERROR] Default detected - Code : %d\n\r",i);
            }
        }
    }
}

void AFE_Results(){
    printf("== Results ==\n\r");
    printf("LED2VAL : %d\n\r",(SPI_Read(0x2A) & 0xFFFFFF));
    printf("ALED2VAL : %d\n\r",(SPI_Read(0x2B) & 0xFFFFFF));
    printf("LED1VAL : %d\n\r",(SPI_Read(0x2C) & 0xFFFFFF));
    printf("ALED1VAL : %d\n\r",(SPI_Read(0x2D) & 0xFFFFFF));
    printf("LED2-ALED2VAL : %d\n\r",(SPI_Read(0x2E) & 0xFFFFFF));
    printf("LED1-ALED1VAL : %d\n\r",(SPI_Read(0x2F) & 0xFFFFFF));
}


void SPI_Write(char address, uint32_t data){
    cs.write(0);
    spi.write(CONTROL0);
    spi.write(0);
    spi.write(0);
    spi.write(0);
    cs.write(1); 
    
    cs.write(0);
    spi.write(address);
    spi.write((data >> 16) & 0xFF);
    spi.write((data >> 8) & 0xFF);
    spi.write(data & 0xFF);
    cs.write(1);   
}

void SPI_Write_Bit(char address, uint8_t bit, bool bit_high){
    unsigned long current_val = SPI_Read(address);
    if (bit_high & !(current_val & 1 << bit)){
        current_val = current_val | (bit_high << bit);
        SPI_Write(address, current_val);
    } else if (!bit_high & (current_val & 1 << bit)) {
        current_val = current_val & ~(bit_high << bit);
        SPI_Write(address, current_val);
    }
}

uint32_t SPI_Read(char address){
    uint32_t data = 0;
    
    cs.write(0);
    spi.write(CONTROL0);
    spi.write(0);
    spi.write(0);
    spi.write(0x1);
    cs.write(1);
    
    cs.write(0);
    spi.write(address);
    data |= (uint32_t)spi.write(0x00) << 16;
    data |= (uint32_t)spi.write(0x00) << 8;
    data |= (uint32_t)spi.write(0x00);
    cs.write(1);
    
    // disable reading from registers
    cs.write(0);
    spi.write(CONTROL0);
    spi.write(0);
    spi.write(0);
    spi.write(0);
    cs.write(1);
    
    return data;
}

bool error_check(int index, uint32_t data_in, uint32_t data_out){
    if (data_in != data_out){
        printf("[ERROR] Address %d : In = %d / Out = %d\n\r",index ,data_in,data_out);
        return 1;
    }else{
        return 0;
    }
}