/*
 * Copyright (c) 2011 Greg Brush
 * Portions copyright (C) 2014 Clark Scheff
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * This code based on:
 *  WiiChuck
 *  http://mbed.org/users/FrankWeissenborn/libraries/WiiChuck/lnae1a
 *  2011-03-14
 *  New initialization compatible with 3rd party controllers
 *
 */
 
#ifndef __WIICHUCK_H
#define __WIICHUCK_H

#include "mbed.h"

#define NUNCHUCK_ADDR           0xA4
#define NUNCHUCK_READLEN        0x06
#define I2C_ACK                 0
#define I2C_READ_DELAY          0.0001
#define I2C_DEFAULT_FREQUENCY   400000

#define JOY_X_IDX               0
#define JOY_Y_IDX               1
#define ACC_X_IDX               2
#define ACC_Y_IDX               3
#define ACC_Z_IDX               4
#define BUTTON_IDX              5

/**
 * @struct nunchuck_data_s
 * @brief Data structure used for holding nunchuck data
 * @var joyX Single byte indicating the position of the joystick in the X axis
 * @var joyY Single byte indicating the position of the joystick in the Y axis
 * @var accX 10-bit value representing the acceleration in the X axis
 * @var accY 10-bit value representing the acceleration in the Y axis
 * @var accZ 10-bit value representing the acceleration in the Z axis
 * @var buttonC Pressed state of the C button
 * @var buttonZ Pressed state of the Z button
 */
typedef struct nunchuck_data_s {
    uint8_t     joyX;
    uint8_t     joyY;
    uint16_t    accX;
    uint16_t    accY;
    uint16_t    accZ;
    bool        buttonC;
    bool        buttonZ;
} nunchuck_data_t;

/**
 * @typedef pt2Func Callback function typedef
 */
typedef void(*pt2Func)(nunchuck_data_t*);

/**
 * Class for interfacing with a Wii Nunchuck (including knock offs)
 *
 * Example:
 * @code
 * // Continuously read data from the nunchuck
 * #include "mbed.h"
 * #include "WiiChuck.h"
 * 
 * WiiChuck nunchuck(PB_9, PB_8);
 * 
 * int main() {
 *     nunchuck_data_t nunchuckData;
 *     
 *     printf("joyX\tjoyY\taccX\taccY\taccZ\tbuttC\tbuttD\r\n"); 
 *     while(1) {
 *         nunchuck.read(&nunchuckData);
 *         printf("%d\t%d\t%d\t%d\t%d\t%s\t%s\r\n",
 *                 nunchuckData.joyX, nunchuckData.joyY, nunchuckData.accX, nunchuckData.accY,
 *                 nunchuckData.accZ, nunchuckData.buttonC ? "X" : "", nunchuckData.buttonZ ? "X" : "");
 *     }
 * }
 * @endcode
 */
class WiiChuck {
public:
    /**
     * @param data I2C data pin
     * @param clk I2C clock pin
     */
    WiiChuck(PinName data, PinName clk, uint32_t i2cFrequency=I2C_DEFAULT_FREQUENCY);
    
    /**
     * Reads a packet of data from the nunchuck and fills in the supplied data structure.
     *
     * @param data Pointer to the data that will be filled in.
     */
    bool read(nunchuck_data_t* data);
    
    /**
     * Attach a callback function that can be called when new data is available.
     *
     * @param callback Pointer to function to be called when data has been read
     * @param readInterval Time interval in seconds to perform successive reads from the nunchuck
     */
    void attach(pt2Func callback, float readInterval);
    
    /**
     * Detach the callback and stop polling the nunchuck
     */
    void detach();
    
    /**
     * Gets the value for the x and y position of the joystick when it is in the neutral position.
     *
     * @param centerX Pointer to a uint8_t that will hold the center x value
     * @param centerY Pointer to a uint8_t that will hold the center y value
     */
    bool calibrateJoystickNeutralPosition(uint8_t* centerX, uint8_t* centerY);

private: 
    void getValues();

    I2C _i2c;
    pt2Func _callback;
    Ticker _getValues; 
    bool _initialized;
    nunchuck_data_t _data;
};

#endif