#ifndef PING_H
#define PING_H

#include "mbed.h"
#include "FPointer.h"

#define DEFAULT_MEASUREMENT_DELAY_US    5000
#define SPEED_OF_SOUND_CM_PER_US        0.034029
#define SPEED_OF_SOUND_IN_PER_US        0.0133972441

/**
 * Class for interfacing with the Parallax Ping))) Ultrasonic Sensor
 *
 * See the Ping)) documentation for more information.
 * http://www.parallax.com/sites/default/files/downloads/28015-PING-Documentation-v1.6.pdf
 */
class Ping {
    public:
        /**
         * @param pin Digital I/O pin used for communicating with the Ping)))
         */
        Ping(PinName pin);
    
        /**
         * Starts a one shot reading.
         *
         * @return True if the reading was successfuly started, false if not.
         */
        bool startReading();
        
        /**
         * Starts a reading.
         *
         * @param continuous Continuous readings if true, one-shot if false.
         * @return True if the reading was successfuly started, false if not.
         */
        bool startReading(bool continuous);

        /**
         * Returns whether the sensor is busy getting a reading
         *
         * @return True if busy, false if not.
         */
        bool isBusy() { return mBusy; }
        
        /**
         * Returns whether the valid reading is available
         *
         * @return True if valid, false if not.
         */
        bool isValid() { return mValid; }
        
        /**
         * Returns the raw reading from the sensor.
         *
         * @return The time in uS that the sound travelled from the sensor to an object.
         */
        uint32_t getRawReading() { return mRawReading; }
        
        /**
         * Sets the delay between continuous readings.
         *
         * @param delay_us Delay between readings in microseconds.
         */
        void setContinuousReadingDelay(uint32_t delay_us) { mDelayBetweenReadings = delay_us; }
        
        /**
         * Gets the distance sound can travel, in centimeters, for a given time in uS.
         *
         * @param time_us Time traveled in microseconds.
         * @return The distance traveled in centimeters.
         */
        static float getDistanceCm(uint32_t time_us) { return time_us * SPEED_OF_SOUND_CM_PER_US; }
        
        /**
         * Gets the distance sound can travel, in inches, for a given time in uS.
         *
         * @param time_us Time traveled in microseconds.
         * @return The distance traveled in inches.
         */
        static float getDistanceIn(uint32_t time_us) { return time_us * SPEED_OF_SOUND_IN_PER_US; }
        
        /**
         * Attach a callback function that will be called when a valid reading is available.
         *
         * @param function Function pointer for callback.
         */
        void attach(uint32_t (*function)(uint32_t) = 0) { mCallback.attach(function); }
        
        /**
         * Attach a class method that will be called when a valid reading is available.
         *
         * @param item Class object that contains the method to call.
         * @param method Method to call.
         */ 
        template<class T> 
        void attach(T* item, uint32_t (T::*method)(uint32_t)) { mCallback.attach(item, method); }
    
    private:
        DigitalInOut mSignalIo;
        InterruptIn mEvent;
        
        bool mBusy;
        bool mValid;
        bool mContinuous;
        uint32_t mDelayBetweenReadings;
        
        Timer mTimer;
        Ticker mMeasureDelayTicker;
        
        uint32_t mRawReading;
        
        FPointer mCallback;
        
        void start();
        void stop();
        void nextMeasurement();
};

#endif
