#include "Transport.h"

/* This table helps with the compution of the CRC8 used in OneWire ROM
 *
 * The theory is that it precomputes all possible states that an eight bit shift register
 * can go to from a certain state when given a certain byte. Computationally and array
 * lookup should be considerably faster than simulating the shift register a bit at a time.
 *
 * Table was sourced from maxim document AN937 that describes the standard.
 */
const unsigned char crc8table[256] = {
    0, 94, 188, 226, 97, 63, 221, 131, 194, 156, 126, 32, 163, 253, 31, 65,
    157, 195, 33, 127, 252, 162, 64, 30, 95, 1, 227, 189, 62, 96, 130, 220,
    35, 125, 159, 193, 66, 28, 254, 160, 225, 191, 93, 3, 128, 222, 60, 98,
    190, 224, 2, 92, 223, 129, 99, 61, 124, 34, 192, 158, 29, 67, 161, 255,
    70, 24, 250, 164, 39, 121, 155, 197, 132, 218, 56, 102, 229, 187, 89, 7,
    219, 133, 103, 57, 186, 228, 6, 88, 25, 71, 165, 251, 120, 38, 196, 154,
    101, 59, 217, 135, 4, 90, 184, 230, 167, 249, 27, 69, 198, 152, 122, 36,
    248, 166, 68, 26, 153, 199, 37, 123, 58, 100, 134, 216, 91, 5, 231, 185,
    140, 210, 48, 110, 237, 179, 81, 15, 78, 16, 242, 172, 47, 113, 147, 205,
    17, 79, 173, 243, 112, 46, 204, 146, 211, 141, 111, 49, 178, 236, 14, 80,
    175, 241, 19, 77, 206, 144, 114, 44, 109, 51, 209, 143, 12, 82, 176, 238,
    50, 108, 142, 208, 83, 13, 239, 177, 240, 174, 76, 18, 145, 207, 45, 115,
    202, 148, 118, 40, 171, 245, 23, 73, 8, 86, 180, 234, 105, 55, 213, 139,
    87, 9, 235, 181, 54, 104, 138, 212, 149, 203, 41, 119, 244, 170, 72, 22,
    233, 183, 85, 11, 136, 214, 52, 106, 43, 117, 151, 201, 74, 20, 246, 168,
    116, 42, 200, 150, 21, 75, 169, 247, 182, 232, 10, 84, 215, 137, 107, 53};
    
// read command

void OWTransport::read(OneWire_Instruction * inst, OneWire_ROM * out){
    inst->network.code = READ_ROM;
    inst->network.inst = &OWTransport::_inst_readsetup;
    inst->network.args = out;
    *out = 0; // clear rom to a valid value (0 does pass CRC8 but is impossible to get as a device)
}

void OWTransport::_inst_readsetup(OneWire * which){
    // setup register
    ((TransportRead *)(which->registers))->read = 0;
    ((TransportRead *)(which->registers))->crc8 = 0;
    
    // setup command
    which->readhandle = &OWTransport::_inst_readhandler;
    which->execute.inst = &OWTransport::_inst_read;
    
    // execute command
    OWTransport::_inst_read(which);
}

void OWTransport::_inst_read(OneWire * which){
    if(((TransportRead *)(which->registers))->read < 64){
        which->op_read();
    }else{
        // check if crc8 check passed (remainder 0), failure is not recoverable
        if(((TransportRead *)(which->registers))->crc8) which->abort(CRC_ERROR);
        else which->endInstruction();
    }
}

void OWTransport::_inst_readhandler(OneWire * which, char bit){
    // store bit
    *((unsigned char *)(which->execute.args))|= bit<<(((TransportRead *)(which->registers))->read++%8);
    
    // every 8 bits (byte) compute CRC8 and move to next byte
    if(!(((TransportRead *)(which->registers))->read % 8)){
        ((TransportRead *)(which->registers))->crc8 = crc8table[((TransportRead *)(which->registers))->crc8 ^ (*((unsigned char *)(which->execute.args)))];
        which->execute.args = (void *)(((unsigned char *)(which->execute.args))+1);
    }
}

// search command

void TransportSearchPersist::clear(){
    rom = 0;
    last = -1;
    done = false;
}

void OWTransport::search(OneWire_Instruction * inst, TransportSearchPersist * search){
    inst->network.code = SEARCH_ROM;
    inst->network.inst = &OWTransport::_inst_searchsetup;
    inst->network.args = search;
}

void OWTransport::_inst_searchsetup(OneWire * which){
    if(((TransportSearchPersist *)(which->execute.args))->done){
        // the search had completed, there is nothing more to do
        which->abort(SUCCESS);
        return;
    }

    // setup register
    ((TransportSearch *)(which->registers))->read = 0;
    ((TransportSearch *)(which->registers))->crc8 = 0;
    ((TransportSearch *)(which->registers))->marker = -1;
    ((TransportSearch *)(which->registers))->bit = 0;
    
    // setup command
    which->readhandle = &OWTransport::_inst_searchhandler;
    which->execute.inst = &OWTransport::_inst_search;
    
    // execute command
    OWTransport::_inst_search(which);
}

void OWTransport::_inst_search(OneWire * which){
    if(((TransportSearch *)(which->registers))->read < 64){
        // state machine, read bit 1 -> read bit 2 and make choice -> send choice -> repeat up to 64 times
        if(((TransportSearch *)(which->registers))->bit>>3 & 0x01){
            // send choosen bit and advance
            if(((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]>>(((TransportSearch *)(which->registers))->read%8) &0x01){
                which->op_send1();
            }else{
                which->op_send0();
            }
            // compute crc8
            if(((TransportSearch *)(which->registers))->read % 8 == 7){
                ((TransportSearch *)(which->registers))->crc8 = crc8table[((TransportSearch *)(which->registers))->crc8 ^ ((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]];
            }
            
            // prepare for next cycle (reset state machine)
            ((TransportSearch *)(which->registers))->read+= 1;
            ((TransportSearch *)(which->registers))->bit = 0;
        }else{
            // read bit 1 and 2 (read function advances state machine)
            which->op_read();
        }
    }else{
        ((TransportSearchPersist *)(which->execute.args))->last = ((TransportSearch *)(which->registers))->marker;
        if(((TransportSearchPersist *)(which->execute.args))->last == -1) ((TransportSearchPersist *)(which->execute.args))->done = true;
        
        // check if crc8 check passed (remainder 0), failure is not recoverable
        if(((TransportSearch *)(which->registers))->crc8) which->abort(CRC_ERROR);
        else which->endInstruction();
    }
}

void OWTransport::_inst_searchhandler(OneWire * which, char bit){
    // store bit
    ((TransportSearch *)(which->registers))->bit|= bit<<(((TransportSearch *)(which->registers))->bit>>2 & 0x01);
    
    // advance state
    if(((TransportSearch *)(which->registers))->bit>>2 & 0x01){
        if((((TransportSearch *)(which->registers))->bit & 0x03) == 3){
            // all slaves were removed or broke?
            ((TransportSearchPersist *)(which->execute.args))->done = true; // any search results that were obtained could be invalid so end search
            which->abort(NO_PRESENCE);
            return;
        }else if((((TransportSearch *)(which->registers))->bit & 0x03) == 0){
            // a conflict bit
            if(((TransportSearch *)(which->registers))->read == ((TransportSearchPersist *)(which->execute.args))->last){
                // already searched 0 bit path, this time send a 1
                ((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]|= 0x01<<(((TransportSearch *)(which->registers))->read%8);
            }else if(((TransportSearch *)(which->registers))->read > ((TransportSearchPersist *)(which->execute.args))->last){
                // a new branch to search
                ((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]&= ~(0x01<<(((TransportSearch *)(which->registers))->read%8));
                ((TransportSearch *)(which->registers))->marker = ((TransportSearch *)(which->registers))->read;
            }else if( !(((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]>>(((TransportSearch *)(which->registers))->read%8) &0x01) ){
                // a previous branch decision
                ((TransportSearch *)(which->registers))->marker = ((TransportSearch *)(which->registers))->read;
            }
        }else{
            // no conflict so write the bit
            if(((TransportSearch *)(which->registers))->bit & 0x01){
                ((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]|= 0x01<<(((TransportSearch *)(which->registers))->read%8);
            }else{
                ((unsigned char *)&(((TransportSearchPersist *)(which->execute.args))->rom))[((TransportSearch *)(which->registers))->read/8]&= ~(0x01<<(((TransportSearch *)(which->registers))->read%8));
            }
        }
        
        // tell next call to send the choice
        ((TransportSearch *)(which->registers))->bit|= 0x08;
    }else{
        // we need to read another bit
        ((TransportSearch *)(which->registers))->bit|= 0x04;
    }
}