 /////////////////////////////
//Main code for Thermal Wake
//Chris Roberts
//4-10-18
///////////////////////////// 
 
#define MULTIPLE_PROBES
#define ARM1_DATA_PIN        p25
#define ARM2_DATA_PIN        p19
#define TAIL_DATA_PIN        p23
#define MAX_PROBES           54
 
#include "mbed.h"
#include "DS1820.h"
#include "ExtendedTimer.h"
#include "SDFileSystem.h"
 


DS1820* probe1[MAX_PROBES];
DS1820* probe2[MAX_PROBES];
DS1820* probeTail[MAX_PROBES];
Serial pc(USBTX, USBRX);
SDFileSystem fs(p5, p6, p7, p8, "fs");
Timer t;
Ticker sampleTime;
Ticker saveTime;
DigitalOut led1(LED1);
DigitalOut led3(p14);
DigitalOut ledOn(p16);
DigitalOut ledError(p15);
bool timeToRead;
bool timeToSave;

void triggerSave()
    {
    timeToSave = true;
    }

void triggerCollection() 
    {
    timeToRead = true; //flag to collect data
    }
    
int main()
{
    //mount filesystem
    bool mountFailure = fs.mount();
    if (mountFailure != 0)
    {
        pc.printf("Failed to mount the SD card.\r\n");
        ledError = 1;
        return -1; //end program with error status
        
    }
    pc.printf("Mounted SD card.\r\n");   
    
    unsigned long long fullName; //DS1820 are 64 bits (a long long)
    FILE* out1 = fopen("/fs/arm1.txt","w");
    FILE* out2 = fopen("/fs/arm2.txt","w");
    FILE* out3 = fopen("/fs/tail.txt","w");
    
    ledOn = 1;
    led3 = 0;
    
    //arm 1
    //initialize probe array to DS1820 objects
    int num_arm1 = 0;
    while(DS1820::unassignedProbe(ARM1_DATA_PIN))
    {
        probe1[num_arm1] = new DS1820(ARM1_DATA_PIN);
        num_arm1++;
        if(num_arm1 == MAX_PROBES)
        {
            break;
        }
    }
    pc.printf("found %d sensors on arm 1 \r\n", num_arm1);
    
    //arm 2
    //initialize probe array to DS1820 objects
    int num_arm2 = 0;
    while(DS1820::unassignedProbe(ARM2_DATA_PIN))
    {
        probe2[num_arm2] = new DS1820(ARM2_DATA_PIN);
        num_arm2++;
        if(num_arm2 == MAX_PROBES)
        {
            break;
        }
    }
    pc.printf("found %d sensors on arm 2\r\n", num_arm2);
    
    //tail
    //initialize probe array to DS1820 objects
    int num_tail = 0;
    while(DS1820::unassignedProbe(TAIL_DATA_PIN))
    {
        probeTail[num_tail] = new DS1820(TAIL_DATA_PIN);
        num_tail++;
        if(num_tail == MAX_PROBES)
        {
            break;
        }
    }
    pc.printf("found %d sensors on tail\r\n", num_tail);
    
    for(int i = 0; i < num_arm1; i++)
    {
        fullName = probe1[i]->whoAmI();
        //print 64-bit ID as a hexadecimal number
        fprintf(out1,"%llX\t", fullName);
    }

    for(int i = 0; i < num_arm2; i++)
    {
        fullName = probe2[i]->whoAmI();
        //print 64-bit ID as a hexadecimal number
        fprintf(out2,"%llX\t", fullName);
    }
    
    for(int i = 0; i < num_tail; i++)
    {
        fullName = probeTail[i]->whoAmI();
        //print 64-bit ID as a hexadecimal number
        fprintf(out3,"%llX\t", fullName);
    }

    fprintf(out1,"\r\n");
    fprintf(out2,"\r\n");
    fprintf(out3,"\r\n");
        
    //write a header row
    fprintf(out1, "Time (s) \t Tempurature (deg C)\r\n");
    fprintf(out2, "Time (s) \t Tempurature (deg C)\r\n");
    fprintf(out3, "Time (s) \t Tempurature (deg C)\r\n");
    
    led3 = 1;

    //start timer and ticker
    t.start();
    sampleTime.attach(&triggerCollection, 10); //write data every __ sec(s)
    timeToRead = true; //collect data at t = 0
    
    timeToSave = false;
    saveTime.attach(&triggerSave, 30);
    
    while (t.read() < (60 * 60 * 4))
    { 
        if(timeToRead == true)
        {
            timeToRead = false; //reset data collection flag
            probe1[0]-> convertTemperature(true, DS1820::all_devices);
            probe2[0]-> convertTemperature(true, DS1820::all_devices);
            probeTail[0]-> convertTemperature(true, DS1820::all_devices);
        
        //print from arm 1
            fprintf(out1, "%3.1f\t", t.read());
            //Get temp from each device
            for(int i = 0; i < num_arm1; i++)
            {
                fprintf(out1, "%3.1f\t",  probe1[i]->temperature());
            }
            fprintf(out1,"\r\n"); 
             
        //print from arm 2 
            fprintf(out2, "%3.1f\t", t.read());
            //Get temp from each device
            for(int i = 0; i < num_arm2  ; i++)
            {
                fprintf(out2, "%3.1f\t",  probe2[i]->temperature());
            }
            fprintf(out2,"\r\n");
            
        //print from tail
            fprintf(out3, "%3.1f\t", t.read());
            //Get temp from each device
            for(int i = 0; i < num_tail; i++)
            {
                fprintf(out3, "%3.1f\t",  probeTail[i]->temperature());
            }
            fprintf(out3,"\r\n"); 
            
            
            if(timeToSave)
            {
                ledError = 1;
                timeToSave = false;
                fclose(out1);
                fclose(out2);
                fclose(out3);
                out1 = fopen("/fs/arm1.txt","a");
                out2 = fopen("/fs/arm2.txt","a");
                out3 = fopen("/fs/tail.txt","a");
                ledError = 0;
            }       
            
        }
    }
    
    //close the file and unmount the file system so the SD card is happy
    fclose(out1);
    fclose(out2);
    fclose(out3);
    fs.unmount();
    //turn on LED to let user know it's safe to remove SD card
    led1 = 1;
}
