/* mbed MAX31850
 Library, for the Dallas (Maxim) 1-Wire Digital Thermometer
 * Copyright (c) 2010, Michael Hagberg Michael@RedBoxCode.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * 
 * 
 * This library has been adapted from the DS1820 library to work with MAX31850 thermocouple sensor
 * Chris Roberts (croberts21@my.whitworth.edu)
 * Craig Russell (crussell21@my.whitworth.edu)
 * 
 * For our application, we modified readAll() (previously convertTemperature) to force a small 
 * wait rather than it being optional, and removed the ability to select a single sensor.
 * 
 * getTemp() (previously temperature) has been updated to deal with the correct number of bits that 
 * the temperature is stored in as well as updated convertions to all four units
 * 
 *  as well as adding a setUnits() function to set the 
 * units that the sensor will return.
 * 
 * We removed some functionality that dealt with parasite power. 
 * 
 *
 */

#ifndef MBED_MAX31850_H
#define MBED_MAX31850_H

#include "mbed.h"
#include "LinkedList.h"

#define C 0 //celsius 
#define K 1 //Kelvin
#define F 2 //fahrenheit
#define R 3 //Rankin

extern Serial pc;

/** MAX31850
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MAX31850.h"
 *
 * MAX31850 probe(DATA_PIN);
 *  
 * int main() {
 *     while(true) {
 *         probe.convertTemperature(true, MAX31850::all_devices); //Start temperature conversion, wait until ready
 *         printf("It is %3.1foC\r\n", probe.temperature());
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class MAX31850
 {
public:
    enum devices{
        this_device,     // command applies to only this device
        all_devices };   // command applies to all devices
    
    enum {
        invalid_conversion = -1000
    };

    /** Create a probe object connected to the specified pins
    * 
    * The probe might either by regular powered or parasite powered. If it is parasite
    * powered and power_pin is set, that pin will be used to switch an external mosfet connecting
    * data to Vdd. If it is parasite powered and the pin is not set, the regular data pin
    * is used to supply extra power when required. This will be sufficient as long as the 
    * number of probes is limitted.
    * 
    * For our application, we have the functions set up to only use external power. The constructor still supports
    * parasite power but the functions to not support it.
    */
    MAX31850 (PinName data_pin, PinName power_pin = NC, bool power_polarity = 0); // Constructor with parasite power pin
  
    ~MAX31850();

    /** Function to see if there are MAX31850
     *  devices left on a pin which do not have a corresponding MAX31850
     *  object
    *
    * @return - true if there are one or more unassigned devices, otherwise false
      */
    static bool unassignedProbe(PinName pin);

    /** This routine will initiate the temperature conversion within
      *  all MAX31850
      *  probes. 
      */
    void readAll();

    /** This function will return the probe temperature. Approximately 10ms per
      * probe to read its scratchpad, do CRC check and convert temperature on the LPC1768.
      *
      * @scratchpad scale, may be either 'c', 'f', 'k', and 'r'
      * @returns temperature for that scale, or MAX31850::invalid_conversion (-1000) if CRC error detected.
      */
    float getTemp(int scale = 4);     

    /** This function returns the 64 bit address associated with a
      * particular DS18B20.
      *
      * Contributed by John M. Larkin (jlarkin@whitworth.edu)
      *
      * @returns unsigned long long
      */
    unsigned long long getId();


    /** This function sets the _units variable to the 
     * users desired temperature units
     */
    void setUnits(int);
    

private:
    bool _parasite_power;
    bool _power_mosfet;
    bool _power_polarity;
    
    static char CRC_byte(char _CRC, char byte );
    static bool onewire_reset(DigitalInOut *pin);
    void match_ROM();
    void skip_ROM();
    static bool search_ROM_routine(DigitalInOut *pin, char command, char *ROM_address);
    static void onewire_bit_out (DigitalInOut *pin, bool bit_data);
    void onewire_byte_out(char data);
    static bool onewire_bit_in(DigitalInOut *pin);
    char onewire_byte_in();
    static bool ROM_checksum_error(char *_ROM_address);
    bool scratchpad_checksum_error();
    void read_scratchpad();
    static bool unassignedProbe(DigitalInOut *pin, char *ROM_address);

    DigitalInOut _datapin;
    DigitalOut _parasitepin;
    
    char _ROM[8];
    char scratchpad[9];
    int _units;
    
    static LinkedList<node> probes;
};
#endif