
#include "mbed.h"

#include "ADCInput.h"
#include "DeviceProperties.h"
#include "MillisecondCounter.h"

#include "PowerController.h"



PowerController::PowerController(void)
    :
        led(LED3),
        acquisitionTimer(this, &PowerController::AcquireData),
        pidTimer(this, &PowerController::ProcessPID)
{
    thread.start(this, &PowerController::ThreadEntry);
}


void PowerController::ThreadEntry(void)
{
    for (;;)
    {
        // wait for an activation
        if (ioControl.ActivateRequested())
        {
            // get a snapshot of the configuration
            deviceConfig = Device.GetDeviceConfig();

            // start acquiring data
            Device.ClearAcquisitionData();
            acquisitionTimestamp = 0;
            outputTarget.Set(0);
            acquisitionTimer.start(deviceConfig.samplePeriodMilliseconds);
            
            led = 1;
            switch (deviceConfig.outputMode)
            {
            case OMRamp:
                ExecuteRamp();
                break;
            case OMConstantVoltage:
                ExecuteConstantVoltage();
                break;
            case OMConstantPower:
                ExecuteConstantPower();
                break;
            default:
                break;
            }
            ioControl.SetOutputVoltage(0);
            ioControl.Deactivate();
            led = 0;

            // pause to get a couple more samples, so that we can show the drop back to
            // zero
            rtos::Thread::wait(deviceConfig.samplePeriodMilliseconds * 2);
            acquisitionTimer.stop();
        }
        
        // let others run
        rtos::Thread::wait(1);
    }
}


void PowerController::ExecuteConstantPower(void)
{
    // on entry we know that we have been told by IOControl that 1) we have
    // gotten an activation request and 2) it has enabled the power supply
    // and waited a certain period for it to be ready
        
    // start our initial time
    unsigned powerStartTime = GetMillisecondCount();
    
    // set the initial output for a period
    ioControl.SetOutputVoltage(deviceConfig.activationStartVoltage);
    unsigned openLoopStart = GetMillisecondCount();
    for (;;)
    {
        rtos::Thread::wait(1);
        if (!ioControl.ActivateRequested())
            return;
        if (GetMillisecondCount() - openLoopStart >= deviceConfig.openLoopDurationMilliseconds)
            break;
    }
        
    // start the PID
    pid.Reset();
    pid.SetPID(deviceConfig.kp, deviceConfig.ki, deviceConfig.kd);
    pid.SetPeriodMilliseconds(deviceConfig.msPIDPeriod);
    pid.SetTarget(deviceConfig.constantPower);
    pid.SetOutputRange(0, MAX_OUTPUT_VOLTAGE);
    pidTimer.start(deviceConfig.msPIDPeriod);
    
    outputTarget.Set(deviceConfig.constantPower);

    // continue with constant power until deactivated
    for (;;)
    {
        double elapsedMS = GetMillisecondCount() - powerStartTime;
        if (elapsedMS > deviceConfig.totalActivationMilliseconds)
            break;
        if (!ioControl.ActivateRequested())
            break;
        rtos::Thread::wait(1);
    }
    pidTimer.stop();
}


void PowerController::ExecuteConstantVoltage(void)
{
    // on entry we know that we have been told by IOControl that 1) we have
    // gotten an activation request and 2) it has enabled the power supply
    // and waited a certain period for it to be ready
        
    // set our initial time
    unsigned powerStartTime = GetMillisecondCount();
    
    // continue with constant voltage until deactivated
    ioControl.SetOutputVoltage(deviceConfig.constantVoltage);
    outputTarget.Set(deviceConfig.constantVoltage);
    for (;;)
    {
        double elapsedMS = GetMillisecondCount() - powerStartTime;
        if (elapsedMS > deviceConfig.totalActivationMilliseconds)
            break;
        if (!ioControl.ActivateRequested())
            break;
        rtos::Thread::wait(1);
    }
    pidTimer.stop();
}


void PowerController::ExecuteRamp(void)
{
    // on entry we know that we have been told by IOControl that 1) we have
    // gotten an activation request and 2) it has enabled the power supply
    // and waited a certain period for it to be ready
        
    // start our initial time
    unsigned powerStartTime = GetMillisecondCount();
    
    // set the initial output for a period
    outputTarget.Set(deviceConfig.activationStartVoltage);
    ioControl.SetOutputVoltage(deviceConfig.activationStartVoltage);
    unsigned openLoopStart = GetMillisecondCount();
    for (;;)
    {
        rtos::Thread::wait(1);
        if (!ioControl.ActivateRequested())
            return;
        if (GetMillisecondCount() - openLoopStart >= deviceConfig.openLoopDurationMilliseconds)
            break;
    }
        
    // start the PID
    pid.Reset();
    pid.SetPID(deviceConfig.kp, deviceConfig.ki, deviceConfig.kd);
    pid.SetPeriodMilliseconds(deviceConfig.msPIDPeriod);
    pid.SetTarget(deviceConfig.rampStartPower);
    pid.SetOutputRange(0, MAX_OUTPUT_VOLTAGE);
    pidTimer.start(deviceConfig.msPIDPeriod);

    outputTarget.Set(deviceConfig.rampStartPower);
    
    // generate the ramp
    unsigned start = GetMillisecondCount();
    for (;;)
    {
        
        double elapsedMS = GetMillisecondCount() - start;
        if (elapsedMS > deviceConfig.rampMilliseconds)
            break;
        if (!ioControl.ActivateRequested())
            break;

        float rampPower = 
            deviceConfig.rampStartPower +
            elapsedMS * (deviceConfig.rampEndPower - deviceConfig.rampStartPower) / deviceConfig.rampMilliseconds
            ;
        pid.SetTarget(rampPower);
        outputTarget.Set(rampPower);
        rtos::Thread::wait(1);
    }
    
    // continue with constant power until deactivated
    pid.SetTarget(deviceConfig.rampEndPower);
    for (;;)
    {
        double elapsedMS = GetMillisecondCount() - powerStartTime;
        if (elapsedMS > deviceConfig.totalActivationMilliseconds)
            break;
        if (!ioControl.ActivateRequested())
            break;
        rtos::Thread::wait(1);
    }
    pidTimer.stop();
}


void PowerController::ProcessPID(void)
{
    pid.AddSample(ADC.GetSensedPower());
    ioControl.SetOutputVoltage(pid.GetOutput());
}


void PowerController::AcquireData(void)
{
    // save a sample
    AcquisitionData data;
    data.timestampMilliseconds = acquisitionTimestamp;
    data.outputVoltage = ioControl.GetOutputVoltage();
    data.iSense = ADC.GetSensedCurrent();
    data.vSense = ADC.GetSensedVoltage();
    data.iRMS = ADC.GetRMSCurrent();
    data.vRMS = ADC.GetRMSVoltage();
    data.deactivateSource = ioControl.GetDeactivationSource();
    data.outputTarget = outputTarget.Get();
    Device.AddAcquisitionSample(data);
    
    // update our timestamp
    acquisitionTimestamp += deviceConfig.samplePeriodMilliseconds;
}


