
#include "DeviceProperties.h"
#include "FullBridgeDriver.h"
#include "MillisecondCounter.h"
#include "SPIDAC.h"

#include "IOControl.h"

// ======================================================================
// class IOControl
//    This class implements a state machine that controls the inputs
//    and outputs associated with activation.  It runs autonomously,
//    such that when an the activate line is asserted it enables the
//    RF output, pauses a certain period, then enters a state where
//    it allows the RF output to be driven.  So a user of this class
//    would do the following:
//       - watch to see if an activation has occured (ActivateRequested())
//       - control the output voltage (SetOutputVoltage())
//       - deactivate (Deactivate())
// ======================================================================

const double OUTPUT_VOLTAGE_RANGE_MAX = 220;

IOControl::IOControl(void)
    :
        hvEn(p26, 1),
        shutdown(p17,0),
        activate(p25, PullUp),
        hvFault(p29, PullUp),
        hvControl(p18),
        updateTimer(this, &IOControl::TimerEntry)
{
    // clear
    activateState = Inactive;
    requestedOutputVoltage = 0.0;
    
    // start our timer
    updateTimer.start(1);
}


bool IOControl::ActivateRequested(void)
{
    return activateState == Active;
}


void IOControl::Deactivate(void)
{
    // we can pretty much go to deactivating anytime... it's our safest state,
    // since not only are we not active, but we also prevent activation for a
    // certain period
    mutex.lock();
    activateState = Deactivating;
    if (deactivateSource == DSNone)
        deactivateSource = DSApplication;
    mutex.unlock();

    UpdateState();
}


double IOControl::GetOutputVoltage(void)
{
    return hvControl * OUTPUT_VOLTAGE_RANGE_MAX;
}


void IOControl::SetOutputVoltage(double outputVoltage)
{
    mutex.lock();
    requestedOutputVoltage = outputVoltage;
    mutex.unlock();

    UpdateState();
}


void IOControl::UpdateState(void)
{
    mutex.lock();
    
    unsigned now = GetMillisecondCount();
    unsigned elapsed = now - stateChangeTime;
        
    // capture and check the inputs
    bool isActivateSwitchOn = !activate;
    //bool isHVFault = !hvFault;
    bool isHVFault = false; // ignore for now, getting issues from noise
    bool activateOn = isActivateSwitchOn && !isHVFault;
    
    switch (activateState)
    {
    case Inactive:
        // switch to Activating on activate
        if (activateOn)
        {
            activateState = Activating;
            stateChangeTime = now;
        }
        break;
        
    case Activating:
        // we can switch to active if our debounce interval has elapsed
        if (activateOn)
        {
            if (elapsed >= 100)
            {
                activateState = PoweringUp;
                stateChangeTime = now;
                deactivateSource = DSNone;
                
                // set the current values for iLim and vLim
                DeviceConfig deviceConfig = Device.GetDeviceConfig();
                PWM.SetState(deviceConfig.pwmFrequency);
                SPIDACSend((uint16_t)(1023 * deviceConfig.vLim), (uint16_t)(1023 * deviceConfig.iLim));
            }
        }
        else
        {
            activateState = Inactive;
        }
        break;
        
    case PoweringUp:
        // switch to active after our post-activate delay has elapsed
        if (activateOn)
        {
            DeviceConfig deviceConfig = Device.GetDeviceConfig();
            if (elapsed >= deviceConfig.postActivationDelayMilliseconds)
            {
                // clear the requested output voltage and force it to be set
                // explicitly
                requestedOutputVoltage = 0;
                activateState = Active;
                stateChangeTime = now;
            }
        }
        else
        {
            activateState = Inactive;
        }
        break;
        
    case Active:
        // start deactivating on activate false
        if (!activateOn)
        {
            activateState = Deactivating;
            stateChangeTime = now;
            
            // record the reason for deactivation
            if (!isActivateSwitchOn)
                deactivateSource = DSSwitch;
            else if (isHVFault)
                deactivateSource = DSHVFault;
        }
        break;
        
    case Deactivating:
        // enforce a dead period after deactivation
        if (elapsed >= 500)
            activateState = Deactivated;
        break;
        
    case Deactivated:
        // don't allow reactivation until we see the footswitch is off
        if (!activateOn)
            activateState = Inactive;
        break;
    }
        
    // set our outputs according to the updated state
    switch (activateState)
    {
    case PoweringUp:
        hvControl = 0;      // output voltage zero
        PWM.Enable(false);  // PWM off
        // TODO... fix this... hvEn is temporarily negated
        hvEn = true;       // power supply active (low)
        shutdown = false;    // shutdown not active (low)
        break;
        
    case Active:
        hvControl = requestedOutputVoltage / OUTPUT_VOLTAGE_RANGE_MAX;      // output voltage as requested
        PWM.Enable(true);  // PWM on
        // TODO... fix this... hvEn is temporarily negated
        hvEn = true;       // power supply active (low)
        shutdown = false;    // shutdown not active (low)
        break;
        
    default:
        hvControl = 0;      // output voltage zero
        PWM.Enable(false);  // PWM off
        // TODO... fix this... hvEn is temporarily negated
        hvEn = false;       // power supply not active (high)
        shutdown = true;    // shutdown active (high)
        break;
    }

    mutex.unlock();
}


void IOControl::TimerEntry(void)
{
    UpdateState();
}

