
#include "mbed.h"

#include "FullBridgeDriver.h"


FullBridgeDriver PWM;

FullBridgeDriver::FullBridgeDriver(void)
{
    // turn on power to PWM
    // on by default LPC_SC->PCONP |= (1<<6);
    isEnabled = false;
    
    // when the pins are used as GPIO I want them to be outputs
    LPC_GPIO2->FIODIR |= (1<<4);
    LPC_GPIO2->FIODIR |= (1<<5);
    
    // and I want them to be low
    LPC_GPIO2->FIOCLR = (1<<4);
    LPC_GPIO2->FIOCLR = (1<<5);
    
    // enable PWM clock, set it to CCLK
    // set to CCLK/4 by default
    LPC_SC->PCLKSEL0 &= ~(3 << 12);
    LPC_SC->PCLKSEL0 |= (1 << 12);
    
    // PWM on
    LPC_SC->PCONP |= (1 << 6);            

    // counter reset
    LPC_PWM1->TCR = 2;                    
    
    // we don't use the prescaler
    LPC_PWM1->PR = 0;     
    
    //reset our counter on match of MR0
    LPC_PWM1->MCR = (1<<1);            

    // enable PWM5 and PWM6, set PWM6 as double edged
    LPC_PWM1->PCR |= 0x6040;

    // set our staring frequency    
    ChangeFrequency(DEFAULT_PWM_FREQUENCY);
    currentFrequency = DEFAULT_PWM_FREQUENCY;
    
    //counter enable, PWM enable
    LPC_PWM1->TCR = (1<<0)|(1<<3);        
}


void FullBridgeDriver::SetState(double kHzFrequency)
{    
    // update our frequency if requested
    if (kHzFrequency != currentFrequency)
    {
        // set the frequency        
        ChangeFrequency(kHzFrequency);
        currentFrequency = kHzFrequency;
    }
}


void FullBridgeDriver::ChangeFrequency(double kHzFrequency)
{
    // calculate the timer period... we want the closest even number
    // so that we can get a 50% duty cycle
    double fTicks = 96000.0 / kHzFrequency;
    double fDoubleTicks = fTicks / 2;
    int doubleTicks = (int)(fDoubleTicks + 0.5);

    // things are set up like this:
    // PWM period: LPC_PWM1->MR0
    // pwm2:
    //     start counter: LPC_PWM1->MR0
    //     stop counter: LPC_PWM1->MR5
    // pwm1:
    //     start counter: LPC_PWM1->MR5
    //     stop counter: LPC_PWM1->MR6
    LPC_PWM1->MR0 = doubleTicks * 2;
    LPC_PWM1->MR5 = doubleTicks;
    LPC_PWM1->MR6 = doubleTicks * 2;
    
    // this latches the values we set above into the shadow registers
    LPC_PWM1->LER = 0x61;
}

double FullBridgeDriver::GetActualFrequency(void)
{
    return 48.0 * 1000 / LPC_PWM1->MR5;
}

void FullBridgeDriver::Enable(bool enabled)
{
    // never mind if we're already in the correct state
    if (enabled == isEnabled)
        return;
    isEnabled = enabled;
        
    // this clears the bit fields for pin selection, which sets them as GPIO
    LPC_PINCON->PINSEL4 &=~(3<<8);            
    LPC_PINCON->PINSEL4 &=~(3<<10);           
    
    // GPIO is what we want if we are disabling
    if (!enabled)
        return;
    
    // else set the bit fields to indicate that our output pins are PWM outputs
    LPC_PINCON->PINSEL4 |= (1<<8);            
    LPC_PINCON->PINSEL4 |= (1<<10);           
}



