
#include "DeviceProperties.h"
#include "FullBridgeDriver.h"

#include "CMSerial.h"

const char VERSION_STRING[] = "0.4";

CMSerial::CMSerial(void)
    :
        led(LED2),
        serial(p13, p14)
{
    lineLength = 0;
    
    serial.baud(115200);
    
    thread.start(this, &CMSerial::ThreadEntry);
}


void CMSerial::ThreadEntry(void)
{
    for (;;)
    {
        char c = serial.getc();
        led = !led;
        
        if (c == '\n')
            continue;
        if (c == '\r')
        {
            line[lineLength] = 0;
            ProcessCommand();
            lineLength = 0;
            continue;
        }
        if (lineLength < sizeof(line) - 1)
            line[lineLength++] = c;
    }
}


void CMSerial::ProcessCommand(void)
{
    if (CommandIs("version"))
    {
        serial.printf("%s\r\n", VERSION_STRING);
        return;
    }
    
    if (CommandIs("ConstantPower"))
    {
        deviceConfig.constantPower = GetNumericArgument(1, deviceConfig.constantPower);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.constantPower);
        return;
    }
    
    if (CommandIs("ConstantVoltage"))
    {
        deviceConfig.constantVoltage = GetNumericArgument(1, deviceConfig.constantVoltage);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.constantVoltage);
        return;
    }
    
    if (CommandIs("Ilim"))
    {
        deviceConfig.iLim = GetNumericArgument(1, deviceConfig.iLim);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.iLim);
        return;
    }
    
    if (CommandIs("Kd"))
    {
        deviceConfig.kd = GetNumericArgument(1, deviceConfig.kd);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.kd);
        return;
    }

    if (CommandIs("Ki"))
    {
        deviceConfig.ki = GetNumericArgument(1, deviceConfig.ki);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.ki);
        return;
    }

    if (CommandIs("Kp"))
    {
        deviceConfig.kp = GetNumericArgument(1, deviceConfig.kp);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.kp);
        return;
    }

    if (CommandIs("OpenLoopDuration"))
    {
        deviceConfig.openLoopDurationMilliseconds = (int)GetNumericArgument(1, deviceConfig.openLoopDurationMilliseconds);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%d\r\n", deviceConfig.openLoopDurationMilliseconds);
        return;
    }

    if (CommandIs("OutputMode"))
    {
        deviceConfig.outputMode = (OutputMode)(int)GetNumericArgument(1, (int)deviceConfig.outputMode);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%d\r\n", deviceConfig.outputMode);
        return;
    }

    if (CommandIs("PostActivationDelay"))
    {
        deviceConfig.postActivationDelayMilliseconds = (int)GetNumericArgument(1, deviceConfig.postActivationDelayMilliseconds);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%d\r\n", deviceConfig.postActivationDelayMilliseconds);
        return;
    }
    
    if (CommandIs("PWMFrequency"))
    {
        // request the value
        PWM.SetState(GetNumericArgument(1, deviceConfig.pwmFrequency));
        
        // update the config to match what we actually got
        deviceConfig.pwmFrequency = PWM.GetActualFrequency();
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.pwmFrequency);
        return;
    }
    
    if (CommandIs("RampDuration"))
    {
        deviceConfig.rampMilliseconds = GetNumericArgument(1, deviceConfig.rampMilliseconds);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.rampMilliseconds);
        return;
    }
    
    if (CommandIs("TotalDuration"))
    {
        deviceConfig.totalActivationMilliseconds = GetNumericArgument(1, deviceConfig.totalActivationMilliseconds);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.totalActivationMilliseconds);
        return;
    }
    
    if (CommandIs("RampEndPower"))
    {
        deviceConfig.rampEndPower = GetNumericArgument(1, deviceConfig.rampEndPower);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.rampEndPower);
        return;
    }
    
    if (CommandIs("RampStartPower"))
    {
        deviceConfig.rampStartPower = GetNumericArgument(1, deviceConfig.rampStartPower);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.rampStartPower);
        return;
    }
    
    if (CommandIs("SamplePeriod"))
    {
        deviceConfig.samplePeriodMilliseconds = GetNumericArgument(1, deviceConfig.samplePeriodMilliseconds);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.samplePeriodMilliseconds);
        return;
    }
    
    if (CommandIs("StartVoltage"))
    {
        deviceConfig.activationStartVoltage = GetNumericArgument(1, deviceConfig.activationStartVoltage);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%f\r\n", deviceConfig.activationStartVoltage);
        return;
    }

    if (CommandIs("status"))
    {
        DeviceConfig config = Device.GetDeviceConfig();
        
        serial.printf("<status");
        serial.printf(" version=\"%s\"", VERSION_STRING);
        serial.printf(" PWMFrequency=\"%lf\"", Device.GetActualPWMFrequency());
        serial.printf(" Kp=\"%lf\"", config.kp);
        serial.printf(" Ki=\"%lf\"", config.ki);
        serial.printf(" Kd=\"%lf\"", config.kd);
        serial.printf(" AcquisitionNumber=\"%d\"", Device.GetAcquisitionNumber());
        serial.printf(" AcquisitionSamples=\"%d\"", Device.GetAcquisitionSampleCount());
        serial.printf(" ConstantPower=\"%f\"", config.constantPower);
        serial.printf(" ConstantVoltage=\"%f\"", config.constantVoltage);
        serial.printf(" OutputMode=\"%d\"", (int)config.outputMode);
        serial.printf(" RampEndPower=\"%lf\"", config.rampEndPower);
        serial.printf(" RampStartPower=\"%lf\"", config.rampStartPower);
        serial.printf(" RampDuration=\"%lf\"", config.rampMilliseconds);
        serial.printf(" SamplePeriod=\"%d\"", config.samplePeriodMilliseconds);
        serial.printf(" StartVoltage=\"%lf\"", config.activationStartVoltage);
        serial.printf(" TotalDuration=\"%lf\"", config.totalActivationMilliseconds);
        serial.printf(" PostActivationDelay=\"%d\"", config.postActivationDelayMilliseconds);
        serial.printf(" OpenLoopDuration=\"%d\"", config.openLoopDurationMilliseconds);
        serial.printf(" Ilim=\"%lf\"", config.iLim);        
        serial.printf(" VIPCalibration=\"%lf\"", config.vipCalibration);
        serial.printf(" Vlim=\"%lf\"", config.vLim);        
        serial.printf("/>\r\n");
        return;
    }
    
    if (CommandIs("GetSample"))
    {
        int acquisition = (int)GetNumericArgument(1, 0);
        int sample = (int)GetNumericArgument(2, 0);
        AcquisitionData sampleData = Device.GetAcquisitionSample(acquisition, sample);
        
        serial.printf("<sample");
        serial.printf(" Time=\"%f\"", 0.001 * sampleData.timestampMilliseconds);
        serial.printf(" Target=\"%f\"", sampleData.outputTarget);
        serial.printf(" OutputVoltage=\"%f\"", sampleData.outputVoltage);
        serial.printf(" Isns=\"%f\"", sampleData.iSense);
        serial.printf(" Vsns=\"%f\"", sampleData.vSense);
        serial.printf(" Irms=\"%f\"", sampleData.iRMS);
        serial.printf(" Vrms=\"%f\"", sampleData.vRMS);
        
        switch (sampleData.deactivateSource)
        {
        case DSApplication:
            serial.printf(" Deact=\"App\"", sampleData.vRMS);
            break;
        case DSHVFault:
            serial.printf(" Deact=\"Fault\"", sampleData.vRMS);
            break;
        case DSSwitch:
            serial.printf(" Deact=\"Switch\"", sampleData.vRMS);
            break;
        default:
            serial.printf(" Deact=\"\"", sampleData.vRMS);
            break;
        }
        
        serial.printf("/>\r\n");
        return;
    }
    
    if (CommandIs("VIPCalibration"))
    {
        deviceConfig.vipCalibration = GetNumericArgument(1, deviceConfig.vipCalibration);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.vipCalibration);
        return;
    }
    
    if (CommandIs("Vlim"))
    {
        deviceConfig.vLim = GetNumericArgument(1, deviceConfig.vLim);
        Device.SetDeviceConfig(deviceConfig);
        serial.printf("%lf\r\n", deviceConfig.vLim);
        return;
    }
    
    serial.printf("Unrecognized command: %s\r\n", line);
}


bool CMSerial::CommandIs(const char *command)
{
    int length = strlen(command);
    if (strncmp(command, line, length) != 0)
        return false;
    if (line[length]!=0 && line[length]!=' ')
        return false;
    return true;
}



double CMSerial::GetNumericArgument(int index, double defaultValue)
{
    // for the moment we just require that arguments be separated by a single space
    const char *start = line;
    for (int i=0; i<index; ++i)
    {
        start = strchr(start, ' ');
        if (start == NULL)
            return defaultValue;
        ++start;
    }
    
    double result;
    if (1 != sscanf(start, "%lf", &result))
        return defaultValue;
    return result;
}


