#ifndef NIMQTT_H
#define NIMQTT_H

#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"

enum { // Message Type
    ZERO,
    CONNECT_NUM,
    CONNACK_NUM,
    PUBLISH_NUM,
    PUBACK_NUM,
    PUBREC_NUM,
    PUBREL_NUM,
    PUBCOMP_NUM,
    SUBSCRIBE_NUM,
    SUBACK_NUM,
    UNSUBSCRIBE_NUM,
    UNSUBACK_NUM,
    PINGREQ_NUM,
    PINGRESP_NUM,
    DISCONNECT_NUM
};

#define CONNECT     CONNECT_NUM     << 4
#define CONNACK     CONNACK_NUM     << 4
#define PUBLISH     PUBLISH_NUM     << 4
#define PUBACK      PUBACK_NUM      << 4
#define PUBREC      PUBREC_NUM      << 4
#define PUBREL      PUBREL_NUM      << 4
#define PUBCOMP     PUBCOMP_NUM     << 4
#define SUBSCRIBE   SUBSCRIBE_NUM   << 4
#define SUBACK      SUBACK_NUM      << 4
#define UNSUBSCRIBE UNSUBSCRIBE_NUM << 4
#define UNSUBACK    UNSUBACK_NUM    << 4
#define PINGREQ     PINGREQ_NUM     << 4
#define PINGRESP    PINGRESP_NUM    << 4
#define DISCONNECT  DISCONNECT_NUM  << 4

enum { // QoS level
    MOST_ONCE_NUM,
    LEAST_ONCE_NUM,
    EXACTLY_ONCE_NUM
};

#define MOST_ONCE       MOST_ONCE_NUM       << 1
#define LEAST_ONCE      LEAST_ONCE_NUM      << 1
#define EXACTLY_ONCE    EXACTLY_ONCE_NUM    << 1

#define KEEP_ALIVE 100 // seconds
#define TIMEOUT 1000 // ms

#define START_THREAD 1

class niMQTT {
    public:
        /** Initialise and launch the MQTT Client
         * \param server the address of your server
         * \param callback a callback to execute on receiving a PUBLISH
         * \param id the id of this client (should be unique)
         * \param port the port of your server
         * \param username your username for the server
         * \param password your password for the server
         * \param debug get a more verbose output
         */
        niMQTT(char *server, void (*callback)(const char *, const char*), char *id="mbed", int port=1883, char *username="", char *password="", bool debug=false);
        void reconnect();
        ~niMQTT();

        /* Publish a message on a topic
         * \param topic the topic
         * \param message the message
         */
        int pub(char *topic, char *message, bool retain=false);

        /** Subscribe to a topic
         * \param topic the topic
         */
        int sub(char *topic, bool unsub=false);

    protected:
        char *server;
        int port;
        char *id;
        void (*callback)(const char *, const char*);
        char *username;
        char *password;

        bool debug;
        bool connected;
        int message_id;

        TCPSocketConnection *socket;

        int remaining_length_length(int remaining_length);
        void get_remaining_length(int remaining_length, char *packet);
        int send(char *packet, int size);

        int decode_remaining_length();
        int recv();

        int init();

        int connect();
        int connack();
        int suback(bool unsub=false);
        int puback();
        int puback_received();
        int pingreq();
        int pingresp();
        int disconnect();

        static void ping_thread_starter(void const *p);
        static void recv_thread_starter(void const *p);
        void ping_thread_worker();
        void recv_thread_worker();
        Thread ping_thread;
        Thread recv_thread;

        void publish_received();
        virtual void call_callback(const char *topic, const char *message);

        bool waiting_new_packet, packet_sent;
        int waiting_connack, waiting_suback, waiting_pingresp;
};

#endif
