 /*
 * Thermistor Temperature library
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "Thermistor.h"


    Thermistor::Thermistor(PinName pin) : _pin(pin) {  // _pin(pin) means pass pin to the AnalogIn constructor
        init();
    }
 
    void Thermistor::init() {
        ThermistorNominal = THERMISTORNOMINAL;
        TemperatureNominal = TEMPERATURENOMINAL;
        BCoefficient = BCOEFFICIENT;
        SeriesResistor = SERIESRESISTOR;
    }
 
    // This is the workhorse routine that calculates the temperature
    // using the Steinhart-Hart equation for thermistors
    // https://en.wikipedia.org/wiki/Steinhart%E2%80%93Hart_equation
    float Thermistor::get_temperature()
    {
        float temperature =0, resistance =0;
        float steinhart =0;
        double a=0;
        int smooth = 5;             // Number of samples to smooth
        
        for(int i=0;i<smooth;i++) {
            a += _pin.read_u16();       // Read 16bit Analog value
        }
        a = a/smooth;                   // Get average of samples    
        //printf("Raw Analog Value for Thermistor = %d\r\n",a);
      
        /* Calculate the resistance of the thermistor from analog votage read. */
        resistance = (float) SeriesResistor / ((65536.0 / a) - 1);
        //printf("Resistance for Thermistor = %f\r\n",resistance);
       
        steinhart = resistance / ThermistorNominal;         // (R/Ro)
        steinhart = log(steinhart);                         // ln(R/Ro)
        steinhart /= BCoefficient;                          // 1/B * ln(R/Ro)
        steinhart += 1.0 / (TemperatureNominal + 273.15);   // + (1/To)
        steinhart = 1.0 / steinhart;                        // Invert
        temperature = steinhart - 273.15;                   // convert to C
    
        return temperature;
    }
     
    void Thermistor::set_ThermistorNominal(float thermnom) {
        ThermistorNominal = thermnom;
    }    
    void Thermistor::set_TemperatureNominal(float tempnom){
        TemperatureNominal = tempnom;
    }    
    void Thermistor::set_BCoefficient(float bcoefficient){
        BCoefficient = bcoefficient;
    }    
    void Thermistor::set_SeriesResistor(float resistor){
        SeriesResistor = resistor;
    }    
    


 
