#ifndef __ST7072I_H__
#define __ST7072I_H__

#include "mbed.h"

/**
 * Device driver for LCD SSCI-014052 (Switch Science SKU#1405),
 * which display controller is ST7032i. 
 * https://www.switch-science.com/catalog/1405/
 *
 * Tested with BLENano.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "st7032i.h"
 *
 * int main(void){
 *     // Creates an I2C object
 *     I2C i2c(I2C_SDA0, I2C_SCL0);
 *
 *     // Creates an display object
 *     St7032i st7032i(&i2c);
 *
 *     // Waits for settling the system.
 *     wait_ms(1000);
 *     
 *     // Initialize the display module.
 *     st7032i.init();
 *   
 *     while(1){
 *         st7032i.puts("Hello, World!!");
 *         // Waits 1 second.
 *         wait(1.0);
 *         // Clears the display
 *         st7032i.clear();
 *         // Waits 0.5 second.
 *         wait(0.5);
 *     }
 * }
 * @endcode
 */
class St7032i
{
public:
    /**
     * Return value from functions.
     */
    typedef enum {
        SUCCESS,            /**< Success */
        ERROR_I2C_NO_ACK,   /**< I2C error with no ACK. */
        INPUT_OUT_OF_RANGE, /**< Given input value is out of range. */
    } Status;

    /**
     * Constructor.
     * @param obj Pointer to an I2C object.
     * @return no return value.
     */
    St7032i(I2C *obj);


    /**
     * Initialize the LCD.
     * @returns
     *     SUCCESS on the device was successfully initialized. 
     *     others on error.
     */     
    Status init();
    
    /**
     * Sets display's contrast.
     * @param val Contrast (0-63)
     * @returns
     *     SUCCESS on the device was successfully set constrast.
     *     others on error.
     */
    Status setContrast(uint8_t val);
    
    /**
     * Puts a character at the current cursor position.
     * @param c A character to be shown.
     * @returns
     *     SUCCESS on the device was successfully set cursor position.
     *     others on error.
     */
    Status putc(char c);    
    
    /**
     * Sets the cursor position.
     * @param col column (0-7)
     * @param row row (0 or 1)
     * @returns
     *     SUCCESS on the device was successfully set cursor position.
     *     others on error.
     */
    Status setCursorPosition(uint8_t col, uint8_t row);
    
    /**
     * Clears display. The instruction execution takes 1.08 ms.
     * @param wait Waits instruction execution in this function if true.
     * @returns
     *     SUCCESS on the device was successfully cleared.
     *     others on error.
     */
    Status clear(bool wait = true);
    
    /**
     * Puts a string from the position (0,0). The characters exceeds the display
     * range will be ignored.
     * @param str Pointer to a string to be shown.
     * @param wrap Automatically move to the second line if true.
     * @returns
     *     SUCCESS on the device put the gievn string successfully.
     *     others on error.
     */
    Status puts(const char *str, bool wrap = true);
    
private:
    /**
     * Holds a pointer to an I2C object.
     */
    I2C *i2c;
    
};

#endif   // __ST7072I_H__
