// ----------------------------------------------------------------------------
// Copyright 2016-2019 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------
#ifndef MBED_TEST_MODE
#include "mbed.h"
#include "kv_config.h"
#include "mbed-cloud-client/MbedCloudClient.h" // Required for new MbedCloudClient()
#include "factory_configurator_client.h"       // Required for fcc_* functions and FCC_* defines
#include "m2mresource.h"                       // Required for M2MResource

#include "mbed-trace/mbed_trace.h"             // Required for mbed_trace_*

// Camera setting
#include "EasyAttach_CameraAndLCD.h"
#include "dcache-control.h"
#include "JPEG_Converter.h"
/**** User Selection *********/
#define VIDEO_PIXEL_HW         (320u)  /* QVGA */
#define VIDEO_PIXEL_VW         (240u)  /* QVGA */
#define JPEG_ENCODE_QUALITY    (75)    /* JPEG encode quality (min:1, max:75 (Considering the size of JpegBuffer, about 75 is the upper limit.)) */
/*****************************/
#define DATA_SIZE_PER_PIC      (2u)
#define FRAME_BUFFER_STRIDE    (((VIDEO_PIXEL_HW * DATA_SIZE_PER_PIC) + 31u) & ~31u)
#define FRAME_BUFFER_HEIGHT    (VIDEO_PIXEL_VW)

uint8_t user_frame_buffer0[FRAME_BUFFER_STRIDE * FRAME_BUFFER_HEIGHT]__attribute((aligned(32)));
uint8_t JpegBuffer[1024 * 32]__attribute((aligned(32)));
DisplayBase Display;
JPEG_Converter Jcu;
// End camera setting

const int LED_ON = MBED_CONF_APP_LED_ON;
const int LED_OFF = (!(MBED_CONF_APP_LED_ON));

// Pointers to the resources that will be created in main_application().
static MbedCloudClient *cloud_client;
static bool cloud_client_running = true;
static NetworkInterface *network = NULL;

// Fake entropy needed for non-TRNG boards. Suitable only for demo devices.
const uint8_t MBED_CLOUD_DEV_ENTROPY[] = { 0xf6, 0xd6, 0xc0, 0x09, 0x9e, 0x6e, 0xf2, 0x37, 0xdc, 0x29, 0x88, 0xf1, 0x57, 0x32, 0x7d, 0xde, 0xac, 0xb3, 0x99, 0x8c, 0xb9, 0x11, 0x35, 0x18, 0xeb, 0x48, 0x29, 0x03, 0x6a, 0x94, 0x6d, 0xe8, 0x40, 0xc0, 0x28, 0xcc, 0xe4, 0x04, 0xc3, 0x1f, 0x4b, 0xc2, 0xe0, 0x68, 0xa0, 0x93, 0xe6, 0x3a };

static M2MResource* m2m_get_res;
static M2MResource* m2m_put_res;
static M2MResource* m2m_post_res;
static M2MResource* m2m_deregister_res;

// Additional resources for camera
static M2MResource* m2m_camera_capture_res;

// Camera functions
bool take_photo(uint8_t *photo_data, uint32_t *photo_data_len) {
    JPEG_Converter::bitmap_buff_info_t buff_info;
    JPEG_Converter::encode_options_t   encode_opt;

    if ((photo_data == NULL) || (photo_data_len == NULL)) {
        return false;
    }

    // Jpeg setting
    buff_info.width              = VIDEO_PIXEL_HW;
    buff_info.height             = VIDEO_PIXEL_VW;
    buff_info.format             = JPEG_Converter::WR_RD_YCbCr422;
    buff_info.buffer_address     = (void *)user_frame_buffer0;
    encode_opt.encode_buff_size  = *photo_data_len;
    encode_opt.input_swapsetting = JPEG_Converter::WR_RD_WRSWA_32_16_8BIT;

    dcache_invalid(photo_data, *photo_data_len);
    JPEG_Converter::jpeg_conv_error_t err = Jcu.encode(&buff_info, photo_data, (size_t *)photo_data_len, &encode_opt);    
    if(err != JPEG_Converter::JPEG_CONV_OK)
    {
        return false;
    }
    return true;
}

bool take_and_send_photo() {
    // Send photo data on button click
    uint32_t photo_data_len = sizeof(JpegBuffer);
    bool result = take_photo(JpegBuffer, &photo_data_len);
    if (result) {
        m2m_camera_capture_res->set_value((const uint8_t *)JpegBuffer, photo_data_len);
    }

    return result;
}

/*
 * Monitor the network connection. If the network is disconnected, LED4 on the board lights.
 */
void monitor() {
    static DigitalOut led(LED4, LED_OFF);
    nsapi_connection_status status = network->get_connection_status();
    if(status != NSAPI_STATUS_GLOBAL_UP) {
        led = LED_ON;
        printf("Network is disconnected. Status = %d.\n", status);
    } else {
        led = LED_OFF;
    }
}


void print_client_ids(void)
{
    printf("Account ID: %s\n", cloud_client->endpoint_info()->account_id.c_str());
    printf("Endpoint name: %s\n", cloud_client->endpoint_info()->internal_endpoint_name.c_str());
    printf("Device Id: %s\n\n", cloud_client->endpoint_info()->endpoint_name.c_str());
}

void button_press() {
    m2m_get_res->set_value(m2m_get_res->get_value_int() + 1);
    printf("User button clicked %d times\n", m2m_get_res->get_value_int());

    if(!take_and_send_photo()) {
        printf("Failed to send photo.");
    }
}

void put_update(const char* /*object_name*/)
{
    printf("Camera shutter triggered. %d\n", (int)m2m_put_res->get_value_int());

    if(!take_and_send_photo()) {
        printf("Failed to send photo.");
    }
}

void execute_post(void* /*arguments*/)
{
    printf("POST executed\n");
}

void deregister_client(void)
{
    printf("Unregistering and disconnecting from the network.\n");
    cloud_client->close();
}

void deregister(void* /*arguments*/)
{
    printf("POST deregister executed\n");
    m2m_deregister_res->send_delayed_post_response();

    deregister_client();
}

void client_registered(void)
{
    printf("Client registered.\n");
    print_client_ids();
    // Turn on LED1 when device is registered.
    DigitalOut led(LED1, LED_ON);
}

void client_unregistered(void)
{
    printf("Client unregistered.\n");
    (void) network->disconnect();
    cloud_client_running = false;
    // Turn off LED1 when device is registered.
    DigitalOut led(LED1, LED_OFF);
}

void client_error(int err)
{
    printf("client_error(%d) -> %s\n", err, cloud_client->error_description());
}

void update_progress(uint32_t progress, uint32_t total)
{
    uint8_t percent = (uint8_t)((uint64_t)progress * 100 / total);
    printf("Update progress = %" PRIu8 "%%\n", percent);
}

int main(void)
{
    int status;

    status = mbed_trace_init();
    if (status != 0) {
        printf("mbed_trace_init() failed with %d\n", status);
        return -1;
    }

    // Mount default kvstore
    printf("Application ready\n");
    status = kv_init_storage_config();
    if (status != MBED_SUCCESS) {
        printf("kv_init_storage_config() - failed, status %d\n", status);
        return -1;
    }

    // Connect with NetworkInterface
    printf("Connect to network\n");
    network = NetworkInterface::get_default_instance();
    if (network == NULL) {
        printf("Failed to get default NetworkInterface\n");
        return -1;
    }
    status = network->connect();
    if (status != NSAPI_ERROR_OK) {
        printf("NetworkInterface failed to connect with %d\n", status);
        // Turn on LED2 when the network initialization failed.
        DigitalOut led(LED2, LED_ON);
        return -1;
    }

    printf("Network initialized, connected with IP %s\n\n", network->get_ip_address());

    // Run developer flow
    printf("Start developer flow\n");
    status = fcc_init();
    if (status != FCC_STATUS_SUCCESS) {
        printf("fcc_init() failed with %d\n", status);
        // Turn on LED3 when Pelion Client initialization failed.
        DigitalOut led(LED3, LED_ON);
        return -1;
    }

    // Inject hardcoded entropy for the device. Suitable only for demo devices.
    (void) fcc_entropy_set(MBED_CLOUD_DEV_ENTROPY, sizeof(MBED_CLOUD_DEV_ENTROPY));
    status = fcc_developer_flow();
    if (status != FCC_STATUS_SUCCESS && status != FCC_STATUS_KCM_FILE_EXIST_ERROR && status != FCC_STATUS_CA_ERROR) {
        printf("fcc_developer_flow() failed with %d\n", status);
        // Turn on LED3 when Pelion Client initialization failed.
        DigitalOut led(LED3, LED_ON);
        return -1;
    }

    printf("Create resources\n");
    M2MObjectList m2m_obj_list;

    // GET resource 3200/0/5501
    m2m_get_res = M2MInterfaceFactory::create_resource(m2m_obj_list, 3200, 0, 5501, M2MResourceInstance::INTEGER, M2MBase::GET_ALLOWED);
    if (m2m_get_res->set_value(0) != true) {
        printf("m2m_get_res->set_value() failed\n");
        return -1;
    }

    // PUT resource 3201/0/5853 (camera trigger resource)
    m2m_put_res = M2MInterfaceFactory::create_resource(m2m_obj_list, 3201, 0, 5853, M2MResourceInstance::INTEGER, M2MBase::GET_PUT_ALLOWED);
    if (m2m_put_res->set_value(0) != true) {
        printf("m2m_led_res->set_value() failed\n");
        return -1;
    }
    if (m2m_put_res->set_value_updated_function(put_update) != true) {
        printf("m2m_put_res->set_value_updated_function() failed\n");
        return -1;
    }

    // POST resource 3201/0/5850
    m2m_post_res = M2MInterfaceFactory::create_resource(m2m_obj_list, 3201, 0, 5850, M2MResourceInstance::INTEGER, M2MBase::POST_ALLOWED);
    if (m2m_post_res->set_execute_function(execute_post) != true) {
        printf("m2m_post_res->set_execute_function() failed\n");
        return -1;
    }

    // POST resource 5000/0/1 to trigger deregister.
    m2m_deregister_res = M2MInterfaceFactory::create_resource(m2m_obj_list, 5000, 0, 1, M2MResourceInstance::INTEGER, M2MBase::POST_ALLOWED);

    // Use delayed response
    m2m_deregister_res->set_delayed_response(true);

    if (m2m_deregister_res->set_execute_function(deregister) != true) {
        printf("m2m_post_res->set_execute_function() failed\n");
        return -1;
    }

    // Camera capture resource 3200/0/4014
    m2m_camera_capture_res = M2MInterfaceFactory::create_resource(m2m_obj_list, 3200, 0, 4014, M2MResourceInstance::STRING, M2MBase::GET_ALLOWED);
    if(m2m_camera_capture_res->set_value(0) != true) {
        printf("m2m_camera_capture_res->set_value() failed.\n");
        return -1;
    }

    // Camera start
    EasyAttach_Init(Display);
    Display.Video_Write_Setting(
        DisplayBase::VIDEO_INPUT_CHANNEL_0,
        DisplayBase::COL_SYS_NTSC_358,
        (void *)user_frame_buffer0,
        FRAME_BUFFER_STRIDE,
        DisplayBase::VIDEO_FORMAT_YCBCR422,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        VIDEO_PIXEL_VW,
        VIDEO_PIXEL_HW
    );
    EasyAttach_CameraStart(Display, DisplayBase::VIDEO_INPUT_CHANNEL_0);

    // Jpeg setting
    Jcu.SetQuality(JPEG_ENCODE_QUALITY);


    printf("Register Pelion Device Management Client\n\n");
    cloud_client = new MbedCloudClient(client_registered, client_unregistered, client_error, NULL, update_progress);
    cloud_client->add_objects(m2m_obj_list);
    cloud_client->setup(network); // cloud_client->setup(NULL); -- https://jira.arm.com/browse/IOTCLT-3114

    // An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
    // This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed
    EventQueue eventQueue;
    Thread t;
    // Start the event queue
    t.start(callback(&eventQueue, &EventQueue::dispatch_forever));

    // Setup the button
    InterruptIn btn(USER_BUTTON0);
    btn.mode(PullUp);

    // The button fall handler is placed in the event queue so it will run in
    // thread context instead of ISR context, which allows safely updating the cloud resource
    btn.fall(eventQueue.event(&button_press));

    // Monitor the network connection periodically.
    Ticker ticker;
    ticker.attach(eventQueue.event(&monitor), 5.0);    // second param is period in seconds

    while(cloud_client_running) {
        int in_char = getchar();
        if (in_char == 'i') {
            print_client_ids(); // When 'i' is pressed, print endpoint info
            continue;
        } else if (in_char == 'r') {
            (void) fcc_storage_delete(); // When 'r' is pressed, erase storage and reboot the board.
            printf("Storage erased, rebooting the device.\n\n");
            wait_us(1000000);
            NVIC_SystemReset();
        } else if (in_char > 0 && in_char != 0x03) { // Ctrl+C is 0x03 in Mbed OS and Linux returns negative number
            button_press(); // Simulate button press
            continue;
        }
        deregister_client();
        break;
    }
    return 0;
}

#endif /* MBED_TEST_MODE */
