#ifndef __LEDUTIL_H__
#define __LEDUTIL_H__

/**
 * A simple class to handle LED.
 *
 * Example (tested on BLENano):
 * @code
 * #include "mbed.h"
 * #include "LEDUtil.h"
 *
 * // LED is connected to P0_19 on BLENano and it lights up when the pin state is high.
 * #define LED_PIN_NAME   P0_19
 *
 * void main() {
 *     LEDUtil led1(P0_19, LEDUtil::HIGH_ON_LOW_OFF);
 *
 *     while(true) {
 *         led1.turnOn();
 *         wait(0.5);
 *         led1.turnOff();
 *         wait(0.5);
 *     }
 * }
 * @endcode
 */
class LEDUtil
{

public:

    /** Polarity of LED's configuration. */
    typedef enum {
        /** The target LED lights up when the pin level is high. */
        HIGH_ON_LOW_OFF,
        /** The target LED lights up when the pin level is low. */
        LOW_ON_HIGH_OFF
    } Polarity;

    /**
     * Constructor.
     *
     * @param targetPinName pin name of the target pin which is coneccted with the target LED.
     * @param targetPinPolarity pin polarity of the target. The default value is POLARITY_HIGH_ON_LOW_OFF
     */
    LEDUtil(PinName targetPinName, Polarity targetPinPolarity = HIGH_ON_LOW_OFF);

    /**
     * Turns on the LED.
     */
    void turnOn();

    /**
     * Turns off the LED.
     */
    void turnOff();

    /**
     * Returns if the LED is on.
     * @return Returns true if the LED is on. Otherwise returns false.
     */
    bool isOn();

    /**
     * Returns if the LED is off.
     * @return Returns true if the LED is off. Otherwise returns false.
     */
    bool isOff();

    /**
     * Get polarity setting.
     * @return polarity setting of this LED configuration.
     */
    Polarity getPolarity();


private:

    /* Holds the target LED's pin name. */
    PinName pinName;

    /* Holds polarity of the target LED pin. */
    Polarity polarity;

    /* State of the LED, i.e. on or off. */
    typedef enum {LED_ON, LED_OFF} LedState;

    /*
     * Sets the LED state.
     * @param state LED state to be set.
     */
    void setLedState(LedState state);

    /*
     * Gets the LED state.
     * @return Returns LED state.
     */
    LedState getLedState();
};

#endif