/* 
 * @author Ken Segler
 * @author ken@earthlcd.com
 * @copyright 2013 Ken Segler
 * @version 1.0
 * @date 
 */

#ifndef _EZLCDLIB_H
#define _EZLCDLIB_H

#include "mbed.h"

#define FIFO    0
#define LIFO    1
#define CLEAR   2

#define DELETE  0
#define ENABLE  1
#define DISABLE 2
#define REDRAW  3

/** Interface to the EarthLCD ezLCD3xx line of smart displays
* Derived class from Serial so that you
* can conveniently printf(), putc(), etc to the display.
*
* Example:
* @code
* #include "mbed.h"
* #include "TSISensor.h"
* #include "ezLCDLib.h"
* #include "MMA8451Q.h"
*
* #define MMA8451_I2C_ADDRESS (0x1d<<1)
*
* enum { BLACK, GRAY, SILVER, WHITE, RED, MAROON, YELLOW, OLIVE, LIME, GREEN, AQUA, TEAL, BLUE, NAVY, FUCHSIA, PURPLE };
* #define arLCD
//InterruptIn ezLCD3Int( PTD7);   //  interrupt instance for touch
//InterruptIn ezLCD3Int( PTA13);   //  interrupt instance for touch
* #ifdef arLCD
* InterruptIn ezLCD3Int( PTD4 );   //  interrupt instance for touch
* DigitalOut _nrst(PTC7);
* ezLCD3 lcd(PTA2, PTA1); //rx,tx
* TSISensor tsi;
* AnalogIn adc1(PTB0);
* MMA8451Q acc(PTE25, PTE24, MMA8451_I2C_ADDRESS);
* #endif
*
* DigitalOut myled(LED_BLUE);
* volatile bool ezLCD3Touch = false;
*
* int color = BLACK;
* int temp;
* int touch=0;
* int adc1Count = 0;
* float adc1Average = 0;
*
* char ezID[20];
* char ezVER[20];
* char adcBuf[20];
* void ezLCD3Isr( void )
* {
    *    ezLCD3Touch = true;
* }
*
* int main()
* {
    *    wait(3);
    *    while(!lcd.sync());
    *    ezLCD3Int.fall(&ezLCD3Isr);
    *    ezLCD3Touch = false;
    *    myled=1;
    *    lcd.cls(BLACK);
    *     lcd.theme(0, 111, 106, 0, 130, 0, 13, 12, 101, 100, 0);
    *    lcd.xy(0,0);//11
    *    lcd.color(BLACK);    //1
    *    lcd.box(319,239, 1 );//2
    *    lcd.color(155);
    *    lcd.xy(1,1);
    *    lcd.line(317,1);
    *    lcd.xy(1,1);
    *    lcd.line(1,237);
    *    lcd.color(WHITE);
    *    lcd.xy(2,237);
    *    lcd.line(317,237);
    *    lcd.line(317,2);
    *    lcd.color(WHITE);//3
    *    lcd.xy(300,100);//4
    *    lcd.xy(300,200);//6
    *    lcd.light(50);//8
    *    lcd.xy(40,10);//9
    *    lcd.print("--==[\\[28m Hello \\[65mMBED\\[28m World \\[3m]==--");//10
    *    lcd.xy(20,100);//11
    *    lcd.color(YELLOW);//12
    *    lcd.setStringID( 2,"Button1");
    *    lcd.setStringID( 3,"Button2");
    *    lcd.button(2,20,40,100,40,1,0,10,0,2);
    *    lcd.button(3,200,40,100,40,1,0,10,0,3);
    *    lcd.getStringID( 65,ezID);
    *    lcd.getStringID( 66,ezVER);
    *    lcd.print(ezID);
    *    lcd.color(PURPLE);
    *    lcd.xy(20,120);//11
    *    lcd.print(ezVER);
    *    lcd.color(TEAL);
    *    lcd.xy(20,140);
    *    lcd.printf("MaxX %d MaxY %d", lcd.getXmax()+1, lcd.getYmax()+1); //13 14
    *    lcd.light(5);//15
    *    lcd.light(100);//16
    *    lcd.image(20,165,"0.gif");
    *    lcd.image(55,165,"1.gif");
    *    lcd.image(90,165,"2.gif");
    *    lcd.image(125,165,"3.gif");
    *    lcd.color(WHITE);//17
    *    lcd.setStringID(4, "meter");
    *    lcd.setStringID(5, "%");
    *    lcd.setStringID(6, "X");
    *    lcd.setStringID(7, "Y");
    *    lcd.setStringID(8, "Z");
    *    lcd.staticText(  4, 200, 210, 100, 25, 8, 0, 4);
    *    lcd.progressBar( 5, 200, 180, 100, 25, 1, 0, 100, 0 , 5);
    *    lcd.progressBar( 6, 200, 90, 100, 25, 1, 0, 100, 0 , 6);
    *    lcd.progressBar( 7, 200, 120, 100, 25, 1, 0, 100, 0 , 7);
    *    lcd.progressBar( 8, 200, 150, 100, 25, 1, 0, 100, 0 , 8);
    *
    *    while(1) {
        *        if(ezLCD3Touch) {
            *            ezLCD3Touch = false;
            *            touch=lcd.wstack(LIFO);
            *            myled = !myled;
            *            lcd.xy(10,210);
            *            lcd.color(BLACK);
            *            lcd.box(180,20,1);
            *            lcd.color(WHITE);
            *            switch( touch ) {
                *                case    240:
                    *                    lcd.print("Button1 Pressed");
                    *                    break;
                *                case    214:
                    *                    lcd.print("Button1 Released");
                    *                    break;
                *                case    224:
                    *                    lcd.print("Button1 Cancel");
                    *                    break;
                *                case    340:
                    *                    lcd.print("Button2 Pressed");
                    *                    break;
                *                case    314:
                    *                    lcd.print("Button2 Released");
                    *                    break;
                *                case    324:
                    *                    lcd.print("Button2 Cancel");
                    *                    break;
                *                default:
                    *                    lcd.print("default");
                    *                    break;
                    *
            }
            *
        }
        *    adc1Average += adc1.read();
        *    adc1Count ++;
        *    if (adc1Count == 500) {
            *    adc1Count=0;
            *    sprintf(adcBuf, "%1.3f V",(adc1Average/500)*3.3);
            *    lcd.setStringID(4, adcBuf);
            *    lcd.wstate(4,REDRAW);
            *    adc1Average =0;
            *
        }
        *    if(adc1Count ==250) {
            *    lcd.wvalue(5,abs( tsi.readPercentage()*100));
            *    lcd.wvalue(6,abs(acc.getAccX())*100);
            *    lcd.wvalue(7,abs(acc.getAccY())*100);
            *    lcd.wvalue(8,abs(acc.getAccZ())*100);
            *
        }
        *
    }
    *
    *
}
* @endcode
*/
class ezLCD3: public Stream
{
public:
    /** Create a new interface to a ezLCD3xx display
     * @param tx -- mbed transmit pin
     * @param rx -- mbed receive pin
     */
    ezLCD3(PinName tx, PinName rx);
    /**
    *
    *
    */
    void Rx_interrupt( void );
    /**
    *
    *
    */
    void itoa(int value, char *sp, int radix);
    /**
    *
    *
    */
    void stripSpace(char *str);
    /**
    *
    *
    */
    bool sync( void );

    /** Send a integer to the display
    * @param[in] i integer to send
    */
    void sendInt( int i );

    /** Get a integer from the display
    * @param[out] str string to put the data in
    *
    */
    int getInt( char *str );

    /** Get a string from the display
    * @param[out] str string to put the data in
    *
    */
    void getString( char *str );

    /** Waits for a CR from display
    *
    */
    bool waitForCR( void );

    /** clear the screen to black
    *
    */
    void cls(void);

    /** clear the screen with background color
    * @param[in] bcolor Background color
    */
    void cls(int bColor);

    /** clear the screen with background color and drawing color
    *
    * @param[in] bColor background color
    * @param[in] fColor drawing color
    */
    void cls(int bColor, int fColor);

    /** Set drawing color
    * @param[in] color color to draw with
    */
    void color( int color );

    /** set x y position for drawing and text
     *
     * @param[in] x is the x coordinate
     * @param[in] y is the y coordinate
     */
    void xy(int x, int y);

    /** set pixel at current x y
     *
     */
    void plot(void);

    /** set pixel in current draw color at x y
     *
     * @param[in] x is the x coordinate
     * @param[in] y is the y coordinate
     */
    void plot(int x, int y);

    /** draw line from current x y to new x y
     *
     * @param[in] x is the x coordinate
     * @param[in] y is the y coordinate
     */
    void line(int x, int y);

    /** Set line drawing type
    * @param[in] type Options: 0 = solid, 1= dotted (1 pixel spacing between dots), 2 = dashed (2 pixel spacing between dashes)
    *
    *
    */
    void lineType( int type );

    /** Set line drawing width
    * @param[in] width in pixels
    *
    *
    */
    void lineWidth( int width );
    /** Draw circle in current color at current x y
     *
     * @param[in] radius diameter of circle
     * @param[in] filled  true for a filled box outline if false
     */
    void circle(int radius, bool filled);

    /** draw a box from current x y of width and heigth
     *
     * @param[in] width
     * @param[in] heigth
     * @param[in] filled  true for a filled box outline if false
     */
    void box(int width, int heigth, bool filled);

    /** set backlight brightness
     *
     * @param[in] level is brightness 0=off 100=full in steps of 32
     */
    void light(int level);

    /** Send a command direct to display
    * @param str command string
    *
    * Example:\n
    * @code
    * lcd.sendCommand( "cls" ); // clear display
    * @endcode
    */
    void sendCommand( char *str );

    /** Return Xmax of display
    * @return int Xmax of display
    *
    */
    int getXmax( void );

    /** Return Ymax of display
    * @return int Ymax of display
    *
    */
    int getYmax( void );

    /** Send a string of data to display
    *
    */
    void sendString( char *str );

    /** Set stringID to display
    * @param ID string ID to write
    * @param str string to write
    *
    */
    void setStringID( int ID, char *str );

    /** Get stringID from Display
    * @param ID string ID to read
    * @param str string to put data in
    *
    * Example:
    * @code
    * lcd.getstringID(66, ezVer);   // get ezLCD Firmware version
    * lcd.print(ezVer);             // print version
    * @endcode
    */
    void getStringID( int ID, char *str );

    /** print string at current x y
    * @param str string prints directly to display
    */
    void print( char *str);

    /** Set widget font
    * Fonts are located in the flash drive of the display\n
    * font 0 and font 1 are builtin fonts and are much faster to draw\n
    * in the /EZUSERS/FONTS and /EZSYS/FONTS directory\n
    * use the ezLCD3xx font convert utilty to convert true type fonts
    * @param str font name
    */
    void fontw( int id, char *str);

    void fontw( int ID, int font);
    /**
    *
    *
    *
    */
    void font(char *str);

    /**
    *
    *
    *
    */
    void fontO( bool dir );

    /**
    *
    *
    *
    */
    void clipArea( int l, int t, int r, int b);

    /**
    *
    *
    *
    */
    void clipEnable( bool enable );

    /** Sets color themes for widgets
    *
    *
    */
    void theme(int ID, int EmbossDkColor, int  EmbossLtColor, int TextColor0, int TextColor1, int TextColorDisabled, int Color0, int Color1, int ColorDisabled, int CommonBkColor, int Fontw);

    /**
    *
    *
    *
    */
    void groupBox( int ID, int x, int y, int width, int height, int options, int theme, int stringID) ;

    /** Analog Meter Widget
    *
    *
    */
    //ameter [ID][x][y][width][height][options][value][min][max][theme][stringID][type]
    void ameter( int ID, int x, int y, int w, int h, int options, int value, int min, int max, int theme, int stringID, int type);

    /** Touchzone Widget
    *
    *
    */
    void touchZone( int ID, int x, int y, int w, int h, bool option);

    /** Button Widget
    *
    *
    */
    void button( int ID, int x, int y, int w, int h, int option, int align, int radius, int theme, int stringID );

    /** Display Bitmap
    * Display Bitmap at current x y \n
    * supports GIF BMP JPG \n
    * images are located in the /EZUSERS/FONTS and /EZSYS/FONTS directory
    * @param str filename
    */
    void image( char *str );

    /** Display Bitmap
    * Display Bitmap at specfied x y \n
    * supports GIF BMP JPG \n
    * images are located in the /EZUSERS/FONTS and /EZSYS/FONTS directory
    * @param x x location to start bitmap
    * @param y y location to start bitmap
    * @param str filename
    */
    void image( int x, int y, char *str );

    /** StaticBox Widget
    *
    *
    *
    */
    void staticText( int ID, int x, int y, int w, int h, int option, int theme, int stringID);

    /** ProgressBar Widget
    * @param ID
    * @param x
    * @param y
    * @param w
    * @param h
    * @param option
    * @param value
    * @param max
    * @param theme
    * @param stringID
    *
    *
    */
    void progressBar(int ID, int x, int y, int w, int h, int option, int value, int max, int theme, int stringID);

    /** Slider Widget
    * @param ID
    * @param x
    * @param y
    * @param w
    * @param h
    * @param option
    * @param range
    * @param res
    * @param value
    * @param theme
    *
    */
    void slider(int ID, int x, int y, int w, int h, int option, int range, int res, int value, int theme);
    /** wstate
    *
    *
    *
    */
    void wstate( int ID, int option );

    /** Wvalue set widget values
    * @param ID widget ID to update
    * @param value widget value to update
    */
    void wvalue( int ID, int value);

    /** Wstack is a 32 level stack of widget events
    * @param type LIFO  - Get the last widget event off the stack
    * @param type FIFO  - Get the first widget event off the stack
    * @param type CLEAR - Clear stack
    * @returns widget id and even touch, release
    */
    unsigned int wstack( int type );

    /** Send a character directly to the  serial interface
     * @param c The character to send to the
     */
    int putc(int c);

    /** Receive a character directly to the serial interface
     * @returns c The character received from the
     */
    int getc();

    /**
     * Numerical values for the EarthSEMPL commands.
     */

    enum Commands {
        Command=             0,     /**< Direct command. */
        Status=              1,
        Clr_Screen=          2,     /**< Clear to provided color. */
        Ping=                3,     /**< Return Pong */
        zBeep=               4,     /**< Beep provided duration
                                      *(frequency fixed) */
        Light=               5,     /**< \c 0 (off) to \c 100 (on) */
        Color=               6,
        eColor_ID=           7,
        Font=                10,    /**< Font number. */
        Fontw=               11,    /**< Font number widget. */
        Font_Orient=         12,    /**< Horizontal or vertical. */
        Line_Width=          13,    /**< 1 or 3. */
        Line_Type=           14,    /**< 1=dot dot 2=dash dash. */
        XY=                  15,    /**< X and Y. */
        StringID=            16,    /**< SID ASCII String or File Name that
                                      * ends with 0. */
        Plot=                17,    /**< Place Pixel at X and Y. */
        Line=               18,    /**< Draw a line to X and Y. */
        Box=                 19,    /**< Draws a Box to X and Y optional
                                      * fill. */
        Circle=             20,    /**< Draws a Circle with Radius optional
                                      * fill */
        Arc=                 21,    /**< Draws an Arc with Radius and Begin
                                      * Angle to End Angle. */
        Pie=                 22,    /**< Draws a Pie figure with Radius and
                                      * Begin Angle to End Angle and fills
                                      * it. */
        Picture=             24,    /**< Places a Picture on display. */
        Print=               25,    /**< Places the string on display which
                                      * ends with 0. */
        Beep_Freq=           26,    /**< Set the beeper frequency. */
        Calibrate=           28,    /**< Calibrate touch screen. */
        zReset=              29,    /**< Reset. */
        Rec_Macro=           30,    /**< Record Macro to flash drive. */
        Play_Macro=          31,    /**< Play Macro. */
        Stop_Macro=          32,    /**< Stop Macro. */
        Pause_Macro=         33,    /**< Pause n msec. */
        Loop_Macro=          34,    /**< Loop on Macro. */
        Speed_Macro=         35,    /**< Set the macro speed. */
        Peri=                36,
        ConfigIO=            37,
        IO=                  38,
        IOG=                 39,
        Security=            40,    /**< Set drive security string. */
        Location=            41,    /**< LID Location Vlaue. */
        Upgrade=             43,
        Parameters=          45,
        ClipEnable=          46,    /**< Set clip Enable. */
        ClipArea=            47,    /**< Set clip area. */
        /* Filesystem operations */
        Comment=             50,
        Fsgetcwd=            51,
        Fschdir=             52,
        Fsmkdir=             53,
        Fsrmdir=             54,
        Fsdir=               55,
        Fscopy=              56,
        Fsrename=            57,
        Fsremove=            58,
        Fsmore=              59,
        Format=              60,    /**< Format Flash Drive if string1 =
                                      * "ezLCD" */
        If=                  61,
        Cmd=                 62,
        /* Widget commands */
        Set_Button=          70,    /**< Widget Button. */
        Set_CheckBox=        71,    /**< Widget Checkbox. */
        Set_Gbox=            72,    /**< Widget Group Box. */
        Set_RadioButton=     73,    /**< Widget Radio Button. */
        Set_DMeter=          74,    /**< Widget Digital Meter. */
        DMeter_Value=        75,    /**< Set DMeter value. */
        Set_AMeter=          76,    /**< Widget Analog Meter. */
        AMeter_Value=        77,    /**< Set AMeter value. */
        AMeter_Color=        78,    /**< Set AMeter color */
        Set_TouchZone=       79,    /**< touch zone       */
        Set_Dial=            80,    /**< Widget RoundDial. */
        Set_Slider=          82,    /**< Widget Slider. */
        Set_Progress=        85,    /**< Widget Progress bar. */
        Progress_Value=      86,    /**< Progress value. */
        Set_StaticText=      87,    /**< Widget Static text. */
        StaticText_Value=    88,    /**< Static text Value. */
        Choice=              89,    /**< Widget get choice. */
        Widget_Theme=        90,    /**< Widget Scheme. */
        Widget_Values=       91,    /**<Widget Values (Slider and Dial in this version).*/
        Widget_State=        92,    /**<Widget State (Button, checkbox, radiobutton in this version).*/
        // no id returns the id of the last touched
        Mode=                98,
        Comport=             99,
        Xmax=                100,   /**< Return Xmax width. */
        Ymax=                101,   /**< Return Ymax height. */
        Wait=                102,   /**< Wait for touch. */
        Waitn=               103,   /**< Wait for no touch. */
        Waitt=               104,   /**< Wait for touch. */
        Threshold=           105,   /**< Touch threshold. */
        Verbose=             106,   /**< Controls the verbose mode. */
        Lecho=               107,   /**< Controls the echo mode. */
        Xtouch=              110,   /**< return touchX. */
        Ytouch=              111,   /**< return touchY. */
        Stouch=              112,   /**< return touchS. */
        Wquiet=              113,
        Wstack=              114,
    };

private:
    Serial _ser;
    virtual int _putc(int c);
    virtual int _getc();
};


#endif