/*
 *  This program displays the temperature in Celsius and Fahrenheit
 *  and allows the user to set a "temperature alarm" that activates
 *  once the the temperature exceeds a temperature value specified by the user.
 *  This program also displays graphical "spirit levels" showing
 *  degrees of rotaion along 3 axis.
 *  The user can cycle between screens using sw2 and sw3 buttons,
 *  set the temperature alarm by pressing up and down on the joystick,
 *  control the volume of the alarm with the pot1 knob, and control the
 *  time value of the wait function by pressing left and right on
 *  the joystick.
 *  screen 1: max temperature alarm
 *  screen 2: temperature in Celsius
 *  screen 3: temperature in Fahrenheit
 *  screen 4: spirit level
 *  screen 5: value for wait function
 *
 */


#include "mbed.h"
#include "LM75B.h"                      /* temperature */
#include "C12832.h"                     /* LCD */
#include "MMA7660.h"                    /* for the accelerometer */
#include "reScale.h"

//DigitalOut r_led (LED1);                /* connections for RGB LED */
//DigitalOut g_led (LED2);
//DigitalOut b_led (LED3);

PwmOut spkr(D6);                        /* speaker */

//DigitalOut led(LED1);                   /* shield LEDs */
DigitalOut red(D5);
DigitalOut blue(PTC12);
DigitalOut green(D9);

InterruptIn sw2_int (PTC6);             /* interrupts for on-board switches */
InterruptIn sw3_int (PTA4);
InterruptIn joy_up (PTB10);
InterruptIn joy_down (PTB11);
InterruptIn joy_left (PTC11);
InterruptIn joy_right (PTC10);
InterruptIn joy_fire (PTB23);
AnalogIn pot1 (A0);
AnalogIn pot2 (A1);

static volatile int sw2_trig;           /* switches triggered? */
static volatile int sw3_trig;
static volatile int joy_up_trig;
static volatile int joy_down_trig;
static volatile int joy_right_trig;
static volatile int joy_left_trig;

/*
 * interrupt handlers
 */
void sw2_interrupt (void)
{
    sw2_trig = 1;
}

void sw3_interrupt (void)
{
    sw3_trig = 1;
}

void joy_up_interrupt (void)
{
    joy_up_trig = 1;
}

void joy_down_interrupt (void)
{
    joy_down_trig = 1;
}

void joy_right_interrupt (void)
{
    joy_right_trig = 1;
}

void joy_left_interrupt (void)
{
    joy_left_trig = 1;
}

// pins
C12832 lcd(D11, D13, D12, D7, D10); /* LCD */
LM75B sensor(D14,D15); /* temperature sensor */
MMA7660 MMA(D14, D15); /* accelerometer */

int main ()
{
    double waitTime = 0.5;                  /* value for wait function */
    int menu = 0;                           /* menu */
    int maxTemp = 30;                       /* value for the temperature alarm */
    double fahrenheit;                      /* for calculating fahrenheit */

    float x, y, z;
    int xInt;
    int yInt;
    int zInt;
        
    sw2_trig = 0;
    sw3_trig = 0;
    joy_up_trig = 0;
    joy_down_trig = 0;
    joy_right_trig = 0;

    sw2_int.mode (PullUp);
    sw2_int.fall (&sw2_interrupt);

    sw3_int.mode (PullUp);
    sw3_int.fall (&sw3_interrupt);

    joy_up.mode (PullUp);
    joy_up.fall (&joy_up_interrupt);

    joy_down.mode (PullUp);
    joy_down.fall (&joy_down_interrupt);

    joy_right.mode (PullUp);
    joy_right.fall (&joy_right_interrupt);

    joy_left.mode (PullUp);
    joy_left.fall (&joy_left_interrupt);

    void tempAlarm();
    void noAlarm();
    void thermomCel(int n, int maxTemp);
    void spiritLev(int xYo, int yYo, int zYo);

    /*
     * main loop
     */
    while (1) {

        x = MMA.x();
        y = MMA.y();
        z = MMA.z();

        float xMul = x * 10;
        float yMul = y * 10;
        float zMul = z * 10;
        
        /*
         * used for graphical spirit levels
         */
        int xInt = static_cast<int>(xMul);
        int yInt = static_cast<int>(yMul);
        int zInt = static_cast<int>(zMul);

        /*
         * change into degrees of rotation
         */
        int zTest = (10 - zInt) * 9;
        if (zTest<0) {
            zTest = 0;
        }
        if (zTest>180) {
            zTest = 180;
        }

        int xTest = (xInt * 9) + 90;
        if (xTest<0) {
            xTest = 0;
        }
        if (xTest>180) {
            xTest = 180;
        }

        int yTest = (yInt * 9) + 90;
        if (yTest<0) {
            yTest = 0;
        }
        if (yTest>180) {
            yTest = 180;
        }

        double tempDouble = static_cast<double>(sensor.temp());
        double fahrenheitPartOne = (tempDouble * 1.8) + 32;         /* convert temperature to fahrenheit value */          
        int tempInt = static_cast<int>(sensor.temp());                    /* convert temperature to int value (used for thermometer display)*/

        /*
         *  blue LED if the temperature is lower than the alarm value
         */
        if(sensor.temp()<maxTemp) {
            noAlarm();
        }

        /*
         *  orange LED if the temperature is lower than the alarm value within 2 degrees
         */
        if(sensor.temp()< maxTemp && sensor.temp()> maxTemp - 2) {
            green = 0 & 0;
            blue = 7 & 4;
            red = 0 & 0;
        }
        /*
         *  alarm triggered if the temperature exceeds the temperature alarm value
         */
        if(sensor.temp()>maxTemp) {
            tempAlarm();
        }

        /*
         *  display graphical thermometer and temperature alarm value
         *  on the first screen
         */
        if(menu==0) {
            lcd.cls();
            lcd.locate(0,3);
            lcd.printf("Max Temp Alarm: %.1d", maxTemp);
            thermomCel(tempInt, maxTemp);
            wait(waitTime);
        }

        /*
         *  display graphical thermometer and temperature in celsius
         *  on the second screen
         */
        if(menu==1) {
            lcd.cls();
            lcd.locate(0,3);
            lcd.printf("Celsius: %.1f\n", sensor.temp());
            thermomCel(tempInt, maxTemp);
            wait(waitTime);
        }

        /*
         *  display graphical thermometer and temperature in fahrenheit
         *  on the third screen
         */
        if(menu==2) {
            lcd.cls();
            lcd.locate(0,3);
            lcd.printf("Fahrenheit: %.1lf\n",fahrenheitPartOne); //changed this to fahrenheitPartOne
            thermomCel(tempInt, maxTemp);
            wait(waitTime);
        }
        
        /*
         *  display spirit level
         *  on the fourth screen
         */
        if(menu==3) {
            lcd.cls();
            lcd.locate(0,3);
            spiritLev(xMul, yMul, zMul);
            lcd.locate(20,0);
            lcd.printf("x: %.1d", xTest);
            lcd.locate(60,0);
            lcd.printf("y: %.1d", yTest);
            lcd.locate(100,0);
            lcd.printf("z: %.1d", zTest);
            wait(waitTime);
        }
        
        /*
         *  display wait time value
         *  on the fifth screen
         */
        if(menu==4) {
            lcd.cls();
            lcd.locate(0,3);
            lcd.printf("Wait time value: %.1f", waitTime);
            wait(waitTime);
        }

        /*
         *  pressing right trigger button cycles up through the screens
         */
        if (sw2_trig) {
            menu++;
            if(menu>4) {
                menu = 4;
            }
            sw2_trig = 0;
        }

        /*
         *  pressing left trigger button cycles down through the screens
         */
        if (sw3_trig) {
            menu--;
            if(menu<0) {
                menu = 0;
            }
            sw3_trig = 0;
        }

        /*
         *  pressing up on the joystick increases the temperature alarm value by 1
         */
        if(joy_up_trig) {
            maxTemp = maxTemp + 1;
            //set max value in here
            joy_up_trig = 0;
        }

        /*
         *  pressing down on the joystick decreases the temperature alarm value by 1
         */
        if(joy_down_trig) {
            maxTemp = maxTemp - 1;
            //set min value in here
            joy_down_trig = 0;
        }

        /*
         *  pressing right on the joystick increases the wait value value by 0.25
         */
        if(joy_right_trig) {
            waitTime = waitTime + 0.25;
            //set max value in here
            joy_right_trig = 0;
        }

        /*
         *  pressing left on the joystick decreases the wait value value by 0.25
         */
        if(joy_left_trig) {
            waitTime = waitTime - 0.25;
            if(waitTime<0.25) {
                waitTime = 0.25;
            }            
            joy_left_trig = 0;
        }

    }

}
/*
 *  turn the LED red and make an alarm noise
 *  pot1 controls the speaker volume
 */
void tempAlarm()
{
    float volume = pot1;

    for (float i=2000.0; i<5000.0; i+=100) {
        spkr.period(5.0/i);
        spkr=volume;
        wait(0.01);
    }

    green = 1;
    blue = 1;
    red = 0;
}

/*
 * turns the LED blue and stops the speaker
 */
void noAlarm()
{
    spkr=0;
    green = 1;
    blue = 0;
    red = 1;
}

/*
 *  display the graphical thermometer
 */
void thermomCel(int n, int maxTemp)
{
    lcd.rect(0, 25, 127, 30, 50);
    lcd.fillrect(60, 25, 60 + n, 30, 50);
    lcd.circle(60, 19, 2, 50);
    lcd.line(60, 25, 60, 23, 50);
    lcd.line(60 + n, 25, 60 + n, 20, 50);
    lcd.line(60 + maxTemp, 25, 60 + maxTemp, 20, 50);
}

/*
 *  display the spirit level
 */
void spiritLev(int xYo, int yYo, int zYo)
{
    lcd.rect(20, 10, 30, 30, 50);
    lcd.fillrect(20, 20, 30, 20 - xYo, 50);
    lcd.rect(60, 10, 70, 30, 50);
    lcd.fillrect(60, 20, 70, 20 - yYo, 50);
    lcd.rect(100, 10, 110, 30, 50);
    lcd.fillrect(100, 30 - (10 - zYo), 110, 30, 50);
    //lcd.fillrect(60, 25, 60 + n, 30, 50);
}
