/**
 * Door lock Mechanism which work similar to locking and unlocking
 * by using an RFID card to scan nfc tag which is publish onto MQTT
 * server by saying in door is locked or unlucked.
 * By Kwame Nyantakyi

 * Use of MQTT client code created by Daniel Knox and Fred Barnes
**/

//
#include "mbed.h"
#include "C12832.h"
#include "MQTTEthernet.h"
#include "MQTTClient.h"

/* speed we talk up the [USB] serial-port at */
#define HOST_SPEED (38400)

// Using Arduino pin notation
C12832 lcd(D11, D13, D12, D7, D10);

static Serial host (USBTX, USBRX);      /* connection to host */

// led display
DigitalOut red_led(D5);
DigitalOut green_led(D9);

// joystick left and right
DigitalIn left(A4);
AnalogIn right(A5);

// sw2 and sw3
InterruptIn sw2_int (PTC6);             /* interrupts for the two on-board switches */
InterruptIn sw3_int (PTA4);

static volatile int sw2_trig;           /* switches triggered? */
static volatile int sw3_trig;

PwmOut spkr(D6);

const char* def  = "Unknown";
const char* stat  = "Unknown";


/* connection stuff for MQTT */
typedef struct S_mqtt_params {
    char *hostname;
    int port;
    char *topic;
    char *clientid;
} mqtt_params_t;

/* configure MQTT with hostname, port, topic and clientid */
static mqtt_params_t mqtt_config = {
hostname: "129.12.44.120"
    ,
    port: 1883,
topic: "unikent/users/kn241/room/status"
    ,
clientid: "kn241"
};

/*
* prints stuffs on the serial host and lcd
*/
static int set_up(void)
{
    host.baud (HOST_SPEED);
    host.printf ("\r\n\r\nEthernet MQTT client from K64F\r\n");

    lcd.cls ();
    lcd.locate (0, 0);
    lcd.printf ("Door Lock Mechanism\n");

    return 0;
}


/*
* Display the status of the connection to MQTT aswell as
* the activity of the user.
*
* Roboust function used to print out format,arg of server
* server connection and user interaction on the serial host
* and lcd
*/
static int display_info(const char *format, ...)
{
    va_list arg;
    int re;

    va_start(arg, format);
    re = host.vprintf (format, arg);
    va_end (arg);
    host.printf ("\r\n");

    lcd.fill (0, 20, 128, 12, 0);
    lcd.locate (0, 15);
    va_start (arg, format);
    lcd.vprintf (format, arg);
    va_end (arg);

    return re;
}

/* plays sounds and sets chars to accepted */
void accepted()
{
    def = "Accepted";
    stat= "Unlocked";
    for (float i=2000.0f; i<10000.0f; i+=100) {
        spkr.period(1.0f/i);
        spkr=0.5;
        wait(0.02);
        spkr = 0.0;
    }
}

/* function which checks  when wrong card is scanned */
void declined()
{
    def = "Rejected";
    stat ="Locked";
    red_led = !left;

    for (float i=2000.0f; i<10000.0f; i+=100) {
        spkr.period(1.0/150.0); // 500hz period
        spkr =0.25;
        wait(.02);
        spkr=0.0;
    }
}

//main code, where the program runs
int main()
{
    const char *ip;
    int re = 0;
    MQTT::Message msg;
    char buf[200];

    set_up();
    display_info("Establishing network...");

    /* Brings up the network interface */
    MQTTEthernet eth = MQTTEthernet ();
    ip = eth.get_ip_address();

    display_info("IP address is: %s", ip ?: "Not found.");

    /* Create Mbed Client Interface */
    MQTT::Client<MQTTEthernet, Countdown> client = MQTT::Client<MQTTEthernet, Countdown> (eth);

    /* Create TCP connection */
    eth.open (eth.getEth());
    re = eth.connect (mqtt_config.hostname, mqtt_config.port);


    if (!re) {
        display_info ("Connected Status: Good");

        /* Wait for a short length of time to allow user to see output messages. */
        Thread::wait(1500);

        MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
        data.keepAliveInterval = 20;
        data.cleansession = 1;
        data.MQTTVersion = 3;
        data.clientID.cstring = mqtt_config.clientid;
        re = client.connect (data);

        if (!re) {
            display_info ("MQTT connected!");

            while (!re) {
                /* clear screen to setup - set up door status*/
                lcd.cls ();
                set_up();

                while(1) {

                    //Depending on button pressed display a colour
                    red_led = !left;
                    green_led = !right;

                    if(right && !left) {
                        accepted();
                    } else if(left && !right) {
                        declined();
                    }
                    display_info("Scan :%s \n", def);
                    sprintf (buf, "%s\n", stat);
                    // QoS 0

                    msg.qos = MQTT::QOS0;
                    msg.retained = false;
                    msg.dup = false;
                    msg.payload = (void*)buf;
                    msg.payloadlen = strlen(buf)+1;

                    re = client.publish (mqtt_config.topic, msg);
                    wait(1.0);


                    if (re) {
                        //if data can't be publish
                        display_info ("MQTT publish failed with %d", re);
                    } else {
                        Thread::wait(1500);
                    }
                }
            }
        } else {
            //
            display_info ("Failed to connect to MQTT");
        }
    } else {
        display_info ("Failed to connect (TCP)");
    }

    // closed ethernet socket
    eth.disconnect();
}