/*
 *  dtest.cpp -- driving test for CO657
 *  Copyright (C) 2015 Fred Barnes, University of Kent  <frmb@kent.ac.uk>
 *  GPL >= 2.0
 */

#include "mbed.h"                       /* stock MBED API */

/* assorted globals */

DigitalOut r_led (LED1);                /* connections for RGB LED */
DigitalOut g_led (LED2);
DigitalOut b_led (LED3);

Serial host (USBTX, USBRX);             /* to-host UART via OpenSDAv2 */

InterruptIn sw2_int (PTC6);             /* interrupts for the two on-board switches */
InterruptIn sw3_int (PTA4);

Ticker rgb_tick;                        /* timer for driving the LED */

static volatile uint16_t counters[3];   /* red, green, blue */
static volatile uint8_t bright[3];

static volatile int sw2_trig;           /* switches triggered? */
static volatile int sw3_trig;


/*
 *  turns offset [0-311] into brightness level [0-255-255-0].
 */
uint8_t brilvl (int offset)
{
    while (offset > 311) {
        offset -= 312;
    }
    if (offset < 52) {
        return (uint8_t)(offset * 5);
    } else if (offset < 156) {
        return 255;
    } else if (offset < 208) {
        return (uint8_t)((207 - offset) * 5);
    }
    return 0;
}


/*
 *  resets the various counters to their initial state
 */
void reset_counters (void)
{
    for (int i=0; i<3; i++) {
        counters[i] = 0;
        bright[i] = brilvl (i * 104);
    }
}


/*
 *  interrupt handlers
 */
void rgb_tick_interrupt (void)
{
    for (int i=0; i<3; i++) {
        counters[i] += (uint16_t)bright[i];
    }
    
    r_led = ((counters[0] >> 8) & 1) ^ 1;
    g_led = ((counters[1] >> 8) & 1) ^ 1;
    b_led = ((counters[2] >> 8) & 1) ^ 1;
    
    for (int i=0; i<3; i++) {
        counters[i] &= 0xff;
    }
}


void sw2_interrupt (void)
{
    sw2_trig = 1;
}


void sw3_interrupt (void)
{
    sw3_trig = 1;
}




/*
 *  start here
 */
int main (void)
{
    int ctr;
    
    host.baud (38400);
    host.printf ("Hello, FRDM-K64F driving-test world!\r\n");
    
    sw2_trig = 0;
    sw3_trig = 0;
    
    reset_counters ();
    
    rgb_tick.attach (&rgb_tick_interrupt, 0.001);
    ctr = 0;
    
    sw2_int.mode (PullUp);
    sw2_int.fall (&sw2_interrupt);
    
    sw3_int.mode (PullUp);
    sw3_int.fall (&sw3_interrupt);
    
    /* and then forever.. */
    for (;;) {
        wait (0.02);
        
        if (sw2_trig) {
            ctr += 104;
            sw2_trig = 0;
        }
        if (sw3_trig) {
            ctr += 208;
            sw3_trig = 0;
        }
        
        ctr++;
        if (ctr > 311) {
            ctr -= 312;
        }
        for (int i=0; i<3; i++) {
            bright[i] = brilvl (ctr + (i * 104));
        }
    }
}

