/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// 13.10.12    initial design
// 25.10.12    add autorefresh of screen
// 25.10.12    add standart font
// 20.12.12    add bitmap graphics

// Assorted changes/updates by Fred Barnes, University of Kent <frmb@kent.ac.uk>
// 31.10.15    minor bugfixes, some drawing optimisation

// optional defines :
// #define debug_lcd  1

#include "C12832.h"
#include "mbed.h"
#include "stdio.h"
#include "Small_7.h"

#define BPP    1       // Bits per pixel


C12832::C12832(PinName mosi, PinName sck, PinName reset, PinName a0, PinName ncs, const char* name)
    : _spi(mosi,NC,sck),_reset(reset),_A0(a0),_CS(ncs),GraphicsDisplay(name)
{
    orientation = 1;
    draw_mode = NORMAL;
    char_x = 0;
    lcd_reset();
}


int C12832::width()
{
    if (orientation == 0 || orientation == 2) return 32;
    else return 128;
}

int C12832::height()
{
    if (orientation == 0 || orientation == 2) return 128;
    else return 32;
}


void C12832::invert(unsigned int o)
{
    if(o == 0) wr_cmd(0xA6);
    else wr_cmd(0xA7);
}


void C12832::set_contrast(unsigned int o)
{
    contrast = o;
    wr_cmd(0x81);      //  set volume
    wr_cmd(o & 0x3F);
}

unsigned int C12832::get_contrast(void)
{
    return(contrast);
}


// write command to lcd controller

void C12832::wr_cmd(unsigned char cmd)
{
    _A0 = 0;
    _CS = 0;
    _spi.write(cmd);
    _CS = 1;
}

// write data to lcd controller

void C12832::wr_dat(unsigned char dat)
{
    _A0 = 1;
    _CS = 0;
    _spi.write(dat);
    _CS = 1;
}

// reset and init the lcd controller

void C12832::lcd_reset()
{

    _spi.format(8,3);                 // 8 bit spi mode 3
    _spi.frequency(20000000);          // 19,2 Mhz SPI clock
    _A0 = 0;
    _CS = 1;
    _reset = 0;                        // display reset
    wait_us(50);
    _reset = 1;                       // end reset
    wait_ms(5);

    /* Start Initial Sequence ----------------------------------------------------*/

    wr_cmd(0xAE);   //  display off
    wr_cmd(0xA2);   //  bias voltage

    wr_cmd(0xA0);
    wr_cmd(0xC8);   //  colum normal

    wr_cmd(0x22);   //  voltage resistor ratio
    wr_cmd(0x2F);   //  power on
    //wr_cmd(0xA4);   //  LCD display ram
    wr_cmd(0x40);   // start line = 0
    wr_cmd(0xAF);     // display ON

    wr_cmd(0x81);   //  set contrast
    wr_cmd(0x17);   //  set contrast

    wr_cmd(0xA6);     // display normal


    // clear and update LCD
    memset(buffer,0x00,512);  // clear display buffer
    copy_to_lcd();
    auto_up = 1;              // switch on auto update
    // dont do this by default. Make the user call
    //claim(stdout);           // redirekt printf to lcd
    locate(0,0);
    set_font((unsigned char*)Small_7);  // standart font
}

// set one pixel in buffer

void C12832::pixel (int x, int y, int color)
{
    if ((x >= 128) || (y >= 32) || (x < 0) || (y < 0)) {
        /* out of bounds */
        return;
    }

    if (draw_mode == NORMAL) {
        if (color == 0) {
            buffer[x + ((y & ~7) << 4)] &= ~(1 << (y & 0x07));  // erase pixel
        } else {
            buffer[x + ((y & ~7) << 4)] |= (1 << (y & 0x07));   // set pixel
        }
    } else { // XOR mode
        if (color == 1) {
            buffer[x + ((y & ~7) << 4)] ^= (1 << (y & 0x07));   // xor pixel
        }
    }
}

/* plots a pixel, but without bounds checking (assumed to be done elsewhere) */
void C12832::pixel_nochk (int x, int y, int colour)
{
    if (draw_mode == NORMAL) {
        if (colour == 0) {
            buffer[x + ((y & ~7) << 4)] &= ~(1 << (y & 0x07));  // erase pixel
        } else {
            buffer[x + ((y & ~7) << 4)] |= (1 << (y & 0x07));   // set pixel
        }
    } else { // XOR mode
        if (colour == 1) {
            buffer[x + ((y & ~7) << 4)] ^= (1 << (y & 0x07));   // xor pixel
        }
    }
}

// update lcd

void C12832::copy_to_lcd(void)
{
    
    int i=0;
    
    //page 0
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB0);      // set page address  0
    _A0 = 1;
    for(i=0; i<128; i++) {
        wr_dat(buffer[i]);
    }

    // page 1
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB1);      // set page address  1
    _A0 = 1;
    for(i=128; i<256; i++) {
        wr_dat(buffer[i]);
    }

    //page 2
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB2);      // set page address  2
    _A0 = 1;
    for(i=256; i<384; i++) {
        wr_dat(buffer[i]);
    }

    //page 3
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB3);      // set page address  3
    _A0 = 1;

    _CS = 0;
    
    for(i=384; i<512; i++) {
        wr_dat(buffer[i]);
    }

}

void C12832::cls (void)
{
    memset (buffer, 0x00, 512);  // clear display buffer
    if (auto_up) {
        copy_to_lcd();
    }
}

/*
 *  based on Foley and van Dam, Computer Graphics Principles and Practice,
 *  Second edition in C, Fig 3.8
 *
 *  Adapted back into C from David Wood's occam implementation (line.occ).
 */
void C12832::line (int x0, int y0, int x1, int y1, int colour)
{
    int dx = x1 - x0;
    int dy = y1 - y0;
    int x, y, a_dx, s_x, a_dy, s_y;
    int d, strt, diag;

    
    if (dx == 0) {
        /* vertical line */
        if (y0 > y1) {
            vline (x0, y1, y0, colour);
        } else {
            vline (x0, y0, y1, colour);
        }
        return;
    }
    if (dy == 0) {
        /* horizontal line */
        if (x0 > x1) {
            hline (x1, y0, x0, colour);
        } else {
            hline (x0, y0, x1, colour);
        }
        return;
    }
    
    if ((x0 >= 128) || (x0 < 0) || (y0 >= 32) || (y0 < 0)) {
        /* starts outside raster, so abort */
        return;
    }
    
    x = x0;
    y = y0;
    if (dx > 0) {
        a_dx = dx;  s_x = 1;
    } else {
        a_dx = -dx; s_x = -1;
    }
    if (dy > 0) {
        a_dy = dy;  s_y = 1;
    } else {
        a_dy = -dy; s_y = -1;
    }
    
    if (a_dx > a_dy) {
        strt = a_dy * 2;            /* straight */
        diag = strt - (2 * a_dx);   /* diagonal */
        d = strt - a_dx;

        while ((a_dx >= 0) && (x >= 0) && (x < 128)) {
            pixel_nochk (x, y, colour);
            x += s_x;
            a_dx--;
            
            if (d <= 0) {
                d += strt;
            } else {
                d += diag;
                y += s_y;
                if ((y < 0) || (y >= 32)) {
                    break;          /* while() */
                }
            }
        }
    } else {
        strt = a_dx * 2;            /* straight */
        diag = strt - (2 * a_dy);   /* diagonal */
        d = strt - a_dy;

        while ((a_dy >= 0) && (y >= 0) && (y < 32)) {
            pixel_nochk (x, y, colour);
            y += s_y;
            a_dy--;
            
            if (d <= 0) {
                d += strt;
            } else {
                d += diag;
                x += s_x;
                if ((x < 0) || (x >= 128)) {
                    break;          /* while() */
                }
            }
        }
    }

    if (auto_up) {
        copy_to_lcd ();
    }
}

/* Note: x1 >= x0 */
void C12832::hline (int x0, int y0, int x1, int colour)
{
    int x;
    int yoff;
    uint8_t ybit;

    if (((x0 < 0) && (x1 < 0)) || ((x0 >= 128) && (x1 >= 128)) || (y0 < 0) || (y0 >= 32)) {
        return;                 /* completely outside */
    }
    if (x0 < 0) {
        x0 = 0;
    } else if (x1 >= 128) {
        x1 = 127;
    }

    yoff = ((y0 & ~7) << 4);
    ybit = (1 << (y0 & 0x07));
    
    if (draw_mode == NORMAL) {
        if (colour == 0) {
            for (x=x0; x<=x1; x++) {
                buffer[x + yoff] &= ~ybit;  // erase pixel
            }
        } else {
            for (x=x0; x<=x1; x++) {
                buffer[x + yoff] |= ybit;   // set pixel
            }
        }
    } else { // XOR mode
        if (colour == 1) {
            for (x=x0; x<=x1; x++) {
                buffer[x + yoff] ^= ybit;   // xor pixel
            }
        }
    }

    if (auto_up) {
        copy_to_lcd ();
    }
}

/* Note: y1 >= y0 */
void C12832::vline (int x0, int y0, int y1, int colour)
{
    if (((y0 < 0) && (y1 < 0)) || ((y0 >= 32) && (y1 >= 32)) || (x0 < 0) || (x0 >= 128)) {
        return;                 /* completely outside */
    }
    if (y0 < 0) {
        y0 = 0;
    } else if (y1 >= 32) {
        y1 = 31;
    }

    if ((y0 & ~0x07) == (y1 & ~0x07)) {
        /* first and last pixels are in the same byte */
        uint8_t ybits = ((1 << ((y1 & 0x07) + 1)) - 1) ^ ((1 << (y0 & 0x07)) - 1);
        int yoff = (y0 & ~7) << 4;      /* same as y1 */
        
        if (draw_mode == NORMAL) {
            if (colour == 1) {
                buffer[x0 + yoff] |= ybits;
            } else {
                buffer[x0 + yoff] &= ~ybits;
            }
        } else {    /* XOR mode */
            buffer[x0 + yoff] ^= ybits;
        }
    } else {
        uint8_t st_ybits = (0xff << (y0 & 0x07));
        uint8_t sp_ybits = ((1 << ((y1 & 0x07) + 1)) - 1);
        int st_yoff = (y0 & ~7) << 4;
        int sp_yoff = (y1 & ~7) << 4;
        
        /* fill in first byte */
        if (draw_mode == NORMAL) {
            if (colour == 1) {
                buffer[x0 + st_yoff] |= st_ybits;
            } else {
                buffer[x0 + st_yoff] &= ~st_ybits;
            }
        } else {    /* XOR mode */
            buffer[x0 + st_yoff] ^= st_ybits;
        }
        for (st_yoff += 128; st_yoff < sp_yoff; st_yoff += 128) {
            /* intervening bytes 0xff */
            if (draw_mode == NORMAL) {
                if (colour == 1) {
                    buffer[x0 + st_yoff] = 0xff;
                } else {
                    buffer[x0 + st_yoff] = 0x00;
                }
            } else {    /* XOR mode */
                buffer[x0 + st_yoff] ^= 0xff;
            }
        }
        /* and the last byte */
        if (draw_mode == NORMAL) {
            if (colour == 1) {
                buffer[x0 + sp_yoff] |= sp_ybits;
            } else {
                buffer[x0 + sp_yoff] &= ~sp_ybits;
            }
        } else {    /* XOR mode */
            buffer[x0 + sp_yoff] ^= sp_ybits;
        }
    }
        
    if (auto_up) {
        copy_to_lcd ();
    }
}

void C12832::rect(int x0, int y0, int x1, int y1, int color)
{

    if (x1 > x0) line(x0,y0,x1,y0,color);
    else  line(x1,y0,x0,y0,color);

    if (y1 > y0) line(x0,y0,x0,y1,color);
    else line(x0,y1,x0,y0,color);

    if (x1 > x0) line(x0,y1,x1,y1,color);
    else  line(x1,y1,x0,y1,color);

    if (y1 > y0) line(x1,y0,x1,y1,color);
    else line(x1,y1,x1,y0,color);

    if(auto_up) copy_to_lcd();
}

void C12832::fillrect(int x0, int y0, int x1, int y1, int color)
{
    int l,i;
    if(x0 > x1) {
        i = x0;
        x0 = x1;
        x1 = i;
    }

    if(y0 > y1) {
        i = y0;
        y0 = y1;
        y1 = i;
    }

    for(l = x0; l<= x1; l ++) {
        vline (l, y0, y1, color);
    }
    if(auto_up) copy_to_lcd();
}



void C12832::circle(int x0, int y0, int r, int color)
{

    int draw_x0, draw_y0;
    int draw_x1, draw_y1;
    int draw_x2, draw_y2;
    int draw_x3, draw_y3;
    int draw_x4, draw_y4;
    int draw_x5, draw_y5;
    int draw_x6, draw_y6;
    int draw_x7, draw_y7;
    int xx, yy;
    int di;
    //WindowMax();
    if (r == 0) {       /* no radius */
        return;
    }

    draw_x0 = draw_x1 = x0;
    draw_y0 = draw_y1 = y0 + r;
    if (draw_y0 < height()) {
        pixel(draw_x0, draw_y0, color);     /* 90 degree */
    }

    draw_x2 = draw_x3 = x0;
    draw_y2 = draw_y3 = y0 - r;
    if (draw_y2 >= 0) {
        pixel(draw_x2, draw_y2, color);    /* 270 degree */
    }

    draw_x4 = draw_x6 = x0 + r;
    draw_y4 = draw_y6 = y0;
    if (draw_x4 < width()) {
        pixel(draw_x4, draw_y4, color);     /* 0 degree */
    }

    draw_x5 = draw_x7 = x0 - r;
    draw_y5 = draw_y7 = y0;
    if (draw_x5>=0) {
        pixel(draw_x5, draw_y5, color);     /* 180 degree */
    }

    if (r == 1) {
        return;
    }

    di = 3 - 2*r;
    xx = 0;
    yy = r;
    while (xx < yy) {

        if (di < 0) {
            di += 4*xx + 6;
        } else {
            di += 4*(xx - yy) + 10;
            yy--;
            draw_y0--;
            draw_y1--;
            draw_y2++;
            draw_y3++;
            draw_x4--;
            draw_x5++;
            draw_x6--;
            draw_x7++;
        }
        xx++;
        draw_x0++;
        draw_x1--;
        draw_x2++;
        draw_x3--;
        draw_y4++;
        draw_y5++;
        draw_y6--;
        draw_y7--;

        if ( (draw_x0 <= width()) && (draw_y0>=0) ) {
            pixel(draw_x0, draw_y0, color);
        }

        if ( (draw_x1 >= 0) && (draw_y1 >= 0) ) {
            pixel(draw_x1, draw_y1, color);
        }

        if ( (draw_x2 <= width()) && (draw_y2 <= height()) ) {
            pixel(draw_x2, draw_y2, color);
        }

        if ( (draw_x3 >=0 ) && (draw_y3 <= height()) ) {
            pixel(draw_x3, draw_y3, color);
        }

        if ( (draw_x4 <= width()) && (draw_y4 >= 0) ) {
            pixel(draw_x4, draw_y4, color);
        }

        if ( (draw_x5 >= 0) && (draw_y5 >= 0) ) {
            pixel(draw_x5, draw_y5, color);
        }
        if ( (draw_x6 <=width()) && (draw_y6 <= height()) ) {
            pixel(draw_x6, draw_y6, color);
        }
        if ( (draw_x7 >= 0) && (draw_y7 <= height()) ) {
            pixel(draw_x7, draw_y7, color);
        }
    }
    if(auto_up) copy_to_lcd();
}

void C12832::fillcircle(int x, int y, int r, int color)
{
    int i,up;
    up = auto_up;
    auto_up = 0;   // off
    for (i = 0; i <= r; i++)
        circle(x,y,i,color);
    auto_up = up;
    if(auto_up) copy_to_lcd();
}

void C12832::setmode(int mode)
{
    draw_mode = mode;
}

void C12832::locate(int x, int y)
{
    char_x = x;
    char_y = y;
}



int C12832::columns()
{
    return width() / font[1];
}



int C12832::rows()
{
    return height() / font[2];
}



int C12832::_putc(int value)
{
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
        if(auto_up) copy_to_lcd();
    }
    return value;
}

void C12832::character(int x, int y, int c)
{
    unsigned int hor,vert,offset,bpl,j,i,b;
    unsigned char* zeichen;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }

    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
    // construct the char into the buffer
    for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                pixel(x+i,y+j,0);
            } else {
                pixel(x+i,y+j,1);
            }

        }
    }

    char_x += w;
}


void C12832::set_font(unsigned char* f)
{
    font = f;
}

void C12832::set_auto_up(unsigned int up)
{
    if(up ) auto_up = 1;
    else auto_up = 0;
}

unsigned int C12832::get_auto_up(void)
{
    return (auto_up);
}

void C12832::print_bm(Bitmap bm, int x, int y)
{
    int h,v,b;
    char d;

    for(v=0; v < bm.ySize; v++) {   // lines
        for(h=0; h < bm.xSize; h++) { // pixel
            if(h + x > 127) break;
            if(v + y > 31) break;
            d = bm.data[bm.Byte_in_Line * v + ((h & 0xF8) >> 3)];
            b = 0x80 >> (h & 0x07);
            if((d & b) == 0) {
                pixel(x+h,y+v,0);
            } else {
                pixel(x+h,y+v,1);
            }
        }
    }

}


