/******************************************************************************
 * DESCRIPTION:
 *   Example code for using CMSIS RTOS Signal.  The main thread
 *   creates three threads.  Two of those threads increment a "count" variable,
 *   while the third thread watches the value of "count".  When "count"
 *   reaches a predefined limit, the waiting thread is signaled by one of the
 *   incrementing threads. The waiting thread "awakens" and then modifies
 *   count. The program continues until the incrementing threads reach
 *   TCOUNT. The main program prints the final value of count.
 * SOURCE: Adapted from example code in "Pthreads Programming", B. Nichols
 *   et al. O'Reilly and Associates.
 * MODIFICATOR: (Simon) CHENG Ye
 * LAST REVISED: 06/JAN/2013
 ******************************************************************************/

#include "mbed.h"
#include "cmsis_os.h"

#define NUM_THREADS  3
#define TCOUNT 10
#define COUNT_LIMIT 12

int     count = 0;
osMutexId count_mutex;
osMutexDef(count_mutex);

struct thread_data
{
    osThreadId id;
    long  my_id;
};

struct thread_data thread_data_array[NUM_THREADS-1];
Serial debug(USBTX, USBRX);

void inc_count(void const *threadarg)
{
    int i;
    long my_id;
    osThreadId id;
    struct thread_data *my_data;
    my_data = (struct thread_data *) threadarg;
    id = my_data->id;
    my_id = my_data->my_id;

    for (i=0; i < TCOUNT; i++) {
        osMutexWait(count_mutex, osWaitForever);
        count++;

        /*
         * Check the value of count and signal waiting thread when condition is
         * reached.  Note that this occurs while mutex is locked.
         */
        if (count == COUNT_LIMIT)
        {
            printf("inc_count(): thread %ld, count = %d  Threshold reached. ",
                    my_id, count);
            osSignalSet(id, 0x1);
            printf("Just sent signal.\n");
        }
        printf("inc_count(): thread %ld, count = %d, unlocking mutex\n",
                my_id, count);
        osMutexRelease(count_mutex);

        /* Do some work so threads can alternate on mutex lock */
        osDelay(1000);
    }
    id = osThreadGetId();
    osThreadTerminate(id);
}

void watch_count(void const *t)
{
    osThreadId id;
    long my_id = (long)t;

    printf("Starting watch_count(): thread %ld\n", my_id);

    /*
     * Lock mutex and wait for signal.  Note that the pthread_cond_wait routine
     * will automatically and atomically unlock mutex while it waits.
     * Also, note that if COUNT_LIMIT is reached before this routine is run by
     * the waiting thread, the loop will be skipped to prevent pthread_cond_wait
     * from never returning.
     */
    osMutexWait(count_mutex, osWaitForever);
    while (count < COUNT_LIMIT)
    {
        osMutexRelease(count_mutex);
        printf("watch_count(): thread %ld Count= %d. Going into wait...\n", my_id,count);
        osSignalWait(0x1, osWaitForever);
        osMutexWait(count_mutex, osWaitForever);
        printf("watch_count(): thread %ld Condition signal received. Count= %d\n", my_id,count);
        printf("watch_count(): thread %ld Updating the value of count...\n", my_id);
        count += 125;
        printf("watch_count(): thread %ld count now = %d.\n", my_id, count);
        osMutexRelease(count_mutex);
    }
    printf("watch_count(): thread %ld Unlocking mutex.\n", my_id);
    //osMutexRelease(count_mutex);
    id = osThreadGetId();
    osThreadTerminate(id);
}

void th0(void const *argument) {watch_count(argument);}
osThreadDef(th0, osPriorityNormal, DEFAULT_STACK_SIZE);

void th1(void const *argument) {inc_count(argument);}
osThreadDef(th1, osPriorityNormal, DEFAULT_STACK_SIZE);

void th2(void const *argument) {inc_count(argument);}
osThreadDef(th2, osPriorityNormal, DEFAULT_STACK_SIZE);

int main(int argc, char *argv[])
{
    debug.baud(57600);
    long t1=1, t2=2, t3=3;
    osThreadId id;

    /* Initialize mutex objects */
    count_mutex = osMutexCreate(osMutex(count_mutex));

    id = osThreadCreate(osThread(th0), (void *)t1);
    thread_data_array[0].id = id;
    thread_data_array[0].my_id = t2;
    thread_data_array[1].id = id;
    thread_data_array[1].my_id = t3;
    osThreadCreate(osThread(th1), (void *)&thread_data_array[0]);
    osThreadCreate(osThread(th2), (void *)&thread_data_array[1]);

    printf ("Main(): Waited and joined with %d threads. Final value of count = %d. Done.\n",
            NUM_THREADS, count);
}
