/**
* Author: Allan Veale
* Date: 27/11/19
* Purpose: Datalog from the active wearable test rig fitted with the first
* realistic (foam tissue) leg
*/

//Both the general mbed header and the test rig bench header are needed 
#include "mbed.h"
#include "bench.h"

 
//Experiment methods
void runFatigueExperiment0(int cycles, float targetkPa,float inflateTimeOut,float deflateTimeOut);

// Create bench object - this is used to control the test rig
Bench leg;

/**
 * Main loop
 */
int main()
{   
    leg.setLoggingFrequency(10); //Set datalogging frequency
    
    /* Two extra columns of data will be recorded in this experiment.
    One for the target pressure, and the other for the number of sit and 
    stand cycles currently completed in the experiment */ 
    string colNames[] = {"Target pressure (kPa)","Cycle"}; //add data headings
    leg.setExtraColumns(colNames,2);
    
    float targetP = 350; //Target pressure in kPa
    int expCycles = 50; //Number of sit to stand to sit cycles 
    float vals[] = {targetP,0}; //set initial values of data that will be logged
    leg.setExtraData(vals);
    
    /* Setup all peripherals on rig, display info about SD card and the 
    user interface menu */ 
    leg.initialise();
      
    /*Run an experiment when the button is pressed to start datalogging and 
    stop it if the button is pressed again to stop datalogging 
    (or when experiment stops - then datalogging stops by itself) */
    while (true) {
        if (leg.isLogging()) {
            runFatigueExperiment0(expCycles,targetP,10,10);            
        }
        wait(0.5);
    }
}

/**
 * Shows how a fatigue experiment works. This experiment pressurises the leg to 
 * pressure targetkPa, depressurises it, and then repeats the process cycles 
 * number of times
 * @param cycles: the number of cycles the leg goes up and down
 * @param targetkPa: the pressure at which the valve is opened to let the leg go down
 * @param inflateTimeOut: time in seconds after which experiment will stop if target pressure is not reached
  * @param inflateTimeOut: time in seconds after which experiment will stop if deflation pressure is not reached
 */
void runFatigueExperiment0(int cycles, float targetkPa, float inflateTimeOut,float deflateTimeOut) 
{
    Timer flowT;//used to time flow into and out of actuator
    float loopTime = 0.1; //(s) time between checking pressure
    
    leg.StartLogging();
    
    //Stop the Bench class from printing, so this method can print
    leg.pausePrint();
        
    // Pressurise and depressurise the leg cycles number of times
    for (int i=0; i<cycles; i++) {
        //Print the progress
        leg.pc.printf("\r\nCycle: \t%i out of \t%i",i+1,cycles);

        //Update cycles logged
        float data[] = {targetkPa,i+1};
        leg.setExtraData(data);        

        //Pressurise
        leg.setValve(true);
        flowT.reset();
        flowT.start();// start inflation timer

        //Wait until measured pressure reaches target pressure
        while(leg.getPressure0()*100 < targetkPa && flowT.read() < inflateTimeOut) {

            //Keep checking logging is going
            if (!leg.isLogging()) {
                leg.pc.printf("\r\nExit A");
                //Logging stopped
                leg.setValve(false); //Depressurise
                leg.StopLogging(); //Stop logging data
                leg.resumePrint(); //Let the Bench class print
                return;
            }
            
            leg.LogData();
            wait(loopTime);//Wait a bit
        }
        leg.pc.printf("\r\nTimer inflate: \t%7.2f",flowT.read());
        if(flowT.read() >= inflateTimeOut) {
            leg.pc.printf("\r\nExit B");
            //Logging stopped
            leg.setValve(false); //Depressurise
            leg.StopLogging(); //Stop logging data
            leg.resumePrint(); //Let the Bench class print
            return;
        }
        
        //Depressurise
        leg.setValve(false);
        flowT.reset();

        /*Wait until depressurised (completely depressurised is
        around 10-12 kPa due to current sensor calibration)*/
        while(leg.getPressure0()*100 > 15 && flowT.read() < deflateTimeOut) {
            
            //Keep checking logging is going
            if (!leg.isLogging()) {
                leg.pc.printf("\r\nExit C");
                //Logging stopped
                leg.setValve(false); //Depressurise
                leg.StopLogging(); //Stop logging data
                leg.resumePrint(); //Let the Bench class print
                return;
            }
                        
            leg.LogData();
            wait(loopTime);//Wait a bit
        }
        
        leg.pc.printf("\r\nTimer deflate: \t%7.2f",flowT.read());
        if(flowT.read() >= deflateTimeOut) {
            leg.pc.printf("\r\nExit D");
            //Logging stopped
            leg.setValve(false); //Depressurise
            leg.StopLogging(); //Stop logging data
            leg.resumePrint(); //Let the Bench class print
            return;
        }
    }
    leg.pc.printf("\r\nExit E");
    // Logging stopped as experiment is fully completed
    leg.setValve(false); //Depressurise
    leg.StopLogging(); //Stop logging data
    leg.resumePrint(); //Let the Bench class print    
}

