/*  Copyright 2020 Allan Joshua Veale
 
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
 
        http://www.apache.org/licenses/LICENSE-2.0
 
    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#ifndef _SPTEPSensor_H_
#define _SPTEPSensor_H_

#include "mbed.h"
/**
 * Reading and nulling the SPTE pressure sensor made by festo (analog input)
 * Based on Martijn Grootens' LCM101 class
 */

class SPTEPressureSensor {
public:

    /**
     * @param pin_a_in: PinName of analog input
     * @param offset: of analog value read from the sensor(calibration data)
     * @param factor: multiplication factor for analog value read from the sensor(calibration data)
     */
    SPTEPressureSensor(PinName pin_a_in, float offset, float factor) :
        analog_in_(pin_a_in),
        kOffset_(offset),
        kFactor_(factor)
    {
    }

    /**
     * @return unscaled analog input value
     */
    float getPressureRaw() 
    {
        return analog_in_.read();
    }

    /**
     * @return force value kOffset_ + kFactor_ * getPressureRaw();
     */
    float getPressure() 
    {
        return kOffset_ + kFactor_ * getPressureRaw();
    }
    
    /**
     * sets pressure scaling offset so that current output is pressure zero
     */
    void nullPressure()
    {
        kOffset_ = kOffset_ - getPressure();
        return;        
    }
    
    /**
     * @return offset of analog value
     */
    float get_offset() { return kOffset_; }
    
    /**
     * @return factor multiplication factor for analog value
     */
    float get_factor() { return kFactor_; }


private:
    AnalogIn analog_in_;
    
    float kOffset_;
    const float kFactor_;
};

#endif