/*  Copyright 2020 Allan Joshua Veale
 
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
 
        http://www.apache.org/licenses/LICENSE-2.0
 
    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#ifndef _LINACT_H_
#define _LINACT_H_

#include "mbed.h"
/**
 * Controls a linear actuator for moving the test rig joint up and down
 */

class LinearActuator
{
public:

    /**
     * @param dir_pin: PinName of digital output controlling actuator direction (1 is retract, 0 is extend)
     * @param pwm_pin: PinName of digital pwm output controlling actuator speed
     */
    LinearActuator (PinName dir_pin, PinName pwm_pin) :
        DirPin(dir_pin),PwmPin(pwm_pin)
    {
        PwmPin.period(1/(double)timing::PWMHertz);  // 10kHz PWM
        
        //default to off
        setDirection(false);
        setPWM(0);
    }

    /**
     * @return current direction
     */
    bool getDirection()
    {
        return DirPin.read();
    }

    /**
     * @param dir set current direction (true is retract, false is extend)
     */
    void setDirection(bool dir)
    {
        DirPin.write((int)dir);
        return;
    }

    /**
     * @param DutyCycle is percent of pwm
     */
    void setPWM(int DutyCycle)
    {
        //limit the duty cycle to 0-100%
        if(DutyCycle > 100) {
            DutyCycle = 100;
        } else if (DutyCycle < 0) {
            DutyCycle = 0;
        }
        //convert to a pulse width in seconds
        double onTime = (1/(double)timing::PWMHertz)*((double)DutyCycle/100);
        PwmPin.pulsewidth(onTime);
        return;
    }

    
private:
    DigitalOut DirPin;
    PwmOut PwmPin;
};

#endif