/*  Copyright 2017 Martijn Grootens
 
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
 
        http://www.apache.org/licenses/LICENSE-2.0
 
    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#ifndef _LCM101_H_
#define _LCM101_H_

#include "mbed.h"

/**
 * Simple class to read out an Omega LCM101 S-beam force sensor connected to an analog
 * input.
 */
class Lcm101 {
public:

    /** constructor
     * @param pin_a_in: PinName of analog input
     * @param offset: linear scaling offset for analog value read from sensor(calibration data)
     * @param factor: multiplication factor for analog value read from sensor(calibration data)
     */
    Lcm101(PinName pin_a_in, float offset, float factor) :
        analog_in_(pin_a_in),
        kOffset_(offset),
        kFactor_(factor)
    {
    }

    /**
     * @return unscaled analog input value
     */
    float getForceRaw() 
    {
        return analog_in_.read();
    }

    /**
     * @return force value kOffset_ + kFactor_ * getForceRaw();
     */
    float getForce() 
    {
        return kOffset_ + kFactor_ * getForceRaw();
        //it appears that higher order polynomials show better accuracy, within the 40 kg range that is.
        // Awaiting other instructions, a linear fit will be currently used however.
    }
    
    /**
     * sets force scaling offset so that current output is force zero
     */
    void nullForce()
    {
        kOffset_ = kOffset_ - getForce();
        return;        
    }
    
    /**
     * @return offset of analog value
     */
    float get_offset() { return kOffset_; }
    
    /**
     * @return factor multiplication factor for analog value
     */
    float get_factor() { return kFactor_; }


private:
    AnalogIn analog_in_;
    
    float kOffset_;
    const float kFactor_;
};

#endif