/*  Copyright 2020 Allan Joshua Veale
 
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
 
        http://www.apache.org/licenses/LICENSE-2.0
 
    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

/**
* Author: Allan Veale and Suraj Giri
* Date: 23/10/2020
* Purpose: Datalog from the active wearable test rig fitted with the first
* realistic (foam tissue) leg. Note that the test rig electronics (mbed) needs to be 
* connected with a USB cable to the mini-USB connector closest to the reset button
* The data logging and test rig is controlled with a PC running a serial terminal
* at 115200. One terminal that works well is found at https://www.putty.org/
*/

//Both the general mbed header and the test rig bench header are needed 
#include "mbed.h"
#include "bench.h"
 
//Methods describing the tests
void runFatigueExperiment0(int cycles, float targetkPa, float inflateTimeOut,float deflateTimeOut,int logHz);
void runFailureExperiment0(float targetkPa,int logHz);
void runBenchmarkExperiment0(int logHz);
void runBenchmarkExperiment1(int minForce, float maxAngle, float inflationT, int pwm, int cycles, int logHz);

// Create bench object - this is used to control the test rig
Bench leg;

/**
 * Main loop
 */
int main()
{       
    /* Two extra columns of data will be recorded in this experiment.
    One for the target pressure, and the other for the number of inflation-deflation
    cycles currently completed in the experiment */ 
    string colNames[] = {"Target pressure (kPa)","Cycle"}; //add data headings
    leg.setExtraColumns(colNames,2);
    
    int logHz = 10; //(Hz) datalogging frequency
    float targetP = 300; //Pressure at which inflation is stopped and deflation begins (kPa)  
                                                                                                                                                                                         
    int fatCycles = 40; //Number of inflation cycles for the fatigue test    
    float infTimeAllow = 10;// (s) time allowed for inflation in a fatigue cycle
    float defTimeAllow = 10;// (s) time allowed for deflation in a fatigue cycle
    
    //for benchmark experiment
    float minForce = 5;//(N) force threshold at which leg extension changes to flexion
    float maxAngle = 70;//(deg) angle threshold at which leg flexion changes to extension
    float inflationTime = 3;//(s) time actuator given to inflate before starting experiment
    int pwm = 75; //(%) linear actuator motor pwm
    int benchCycles = 1;//(-) number of flexion/extension cycles
    
    float vals[] = {targetP,0}; //set of initial values of data that will be logged
    leg.setExtraData(vals);
    
    /* Setup all peripherals on rig, display info about SD card and the 
    user interface menu */ 
    leg.initialise();
          
    /*Run an experiment when the button is pressed to start datalogging and 
    stop it if the button is pressed again to stop datalogging 
    (or when experiment stops - then datalogging stops by itself) */
    while (true) {
        if (leg.isLogging()) {//check if logging has started, if so, run an experiment procedure
            leg.pc.printf("Logging started");//user feedback
            
            //uncomment the type of experiment you want to try
            runFatigueExperiment0(fatCycles,targetP,infTimeAllow,defTimeAllow,logHz);
            //runBenchmarkExperiment0(logHz);
            //runBenchmarkExperiment1(minForce, maxAngle, inflationTime, pwm, benchCycles, logHz);
            //runFailureExperiment0(targetP,logHz);        
        }
        wait(0.5);//wait a bit before checking again
    }
}

/**
 * This experiment pressurises the leg to 
 * pressure targetkPa, depressurises it, and then repeats the process cycles 
 * number of times
 * @param cycles: the number of cycles the leg goes up and down
 * @param targetkPa: the pressure at which the valve is opened to let the leg go down (kPa)
 * @param inflateTimeOut: (s) if inflation takes longer than this, experiment is assumed to fail and stops, saving data
 * @param deflateTimeOut: (s) if deflation takes longer than this, experiment is assumed to fail and stops, saving data
 * @param logHz: datalogging frequency
 */
void runFatigueExperiment0(int cycles, float targetkPa, float inflateTimeOut, float deflateTimeOut,int logHz) 
{
    //method constants
    Timer flowT;//timer used to time flow into and out of actuator
        
    float loopTime = 1/logHz; //(s) time between checking pressure
    
    //number of data files to save (so we can create a safety barrier 
    //when lots of cycles are done e.g. you may want 100 cycles in each file 
    //so if something goes wrong, you will have most of the data, to within a 100 cycles)    
    int num_file = 1; 
    
    for (int i=0; i<num_file; i++) {
        //record data
        leg.StartLogging();
        //Stop the Bench class from printing, so this method can print
        leg.pausePrint();

        // Pressurise and depressurise the leg cycles number of times
        for (int i=0; i<cycles; i++) {
            leg.pc.printf("\r\nCycle: \t%i out of \t%i",i+1,cycles);//print the cycle number

            //Update cycles logged
            float data[] = {targetkPa,i+1};
            leg.setExtraData(data);

            //Pressurise
            leg.setValve(true);
            flowT.reset();
            flowT.start();// start inflation timer

            //Wait until measured pressure reaches target pressure or inflation time out is reached
            while((leg.getPressure0()*100 < targetkPa) && (flowT.read() < inflateTimeOut)) {
                //if Logging stopped (by user)
                if (!leg.isLogging()) {                
                    leg.pc.printf("\r\nUser stop during inflation exit");
                    leg.setValve(false); //Depressurise
                    leg.StopLogging(); //Stop logging data
                    leg.resumePrint(); //Let the Bench class print
                    return;//exit method
                }
                leg.LogData();//record datapoint
                wait(loopTime);//Wait a bit
            }

            leg.pc.printf("\r\nTimer inflate: \t%7.2f",flowT.read());
            //if inflation timer times out
            if(flowT.read() >= inflateTimeOut) {
                leg.pc.printf("\r\nInflation timeout exit");
                leg.setValve(false); //Depressurise
                leg.StopLogging(); //Stop logging data
                leg.resumePrint(); //Let the Bench class print
                return;//exit method
            }

            //Depressurise
            leg.setValve(false);
            flowT.reset(); //reset timer (no need to start as it is already going)
            
            /*Wait until depressurised (completely depressurised is
            around 10-12 kPa due to current sensor calibration) or deflation
            timeout is reached*/
            while((leg.getPressure0()*100 > 15) && (flowT.read() < deflateTimeOut)) {
                //if Logging stopped (by user)
                if (!leg.isLogging()) {
                    leg.pc.printf("\r\nUser stop during deflation exit");
                    leg.setValve(false); //Depressurise
                    leg.StopLogging(); //Stop logging data
                    leg.resumePrint(); //Let the Bench class print
                    return;
                }
                leg.LogData();//record a datapoint
                wait(loopTime);//Wait a bit
            }

            leg.pc.printf("\r\nTimer deflate: \t%7.2f",flowT.read());
            //if inflation timer runs out
            if(flowT.read() >= deflateTimeOut) {
                leg.pc.printf("\r\nDeflation timeout exit");
                leg.setValve(false); //Depressurise
                leg.StopLogging(); //Stop logging data
                leg.resumePrint(); //Let the Bench class print
                return;
            }
        }

        /* Logging stopped as experiment is fully completed (desired number of cycles
        reached)*/
        leg.pc.printf("\r\nExit Fatigue Test");
        leg.setValve(false); //Depressurise
        leg.StopLogging(); //Stop logging data
        leg.resumePrint(); //Let the Bench class print
    }
}

/**
 * This experiment pressurises the actuator under test to
 * pressure targetkPa and then depressurises it. It is assumed the user is slowly 
 * increasing the actuator pressure by manually opening up the pressure regulator
 * connected to the actuator. It is also assumed that the user will stop inflation 
 * if they deem the actuator to have failed.
 * @param targetkPa: the pressure at which the valve is opened to deflate the actuator (kPa)
 * @param logHz: datalogging frequency
 */
void runFailureExperiment0(float targetkPa,int logHz) 
{    
    double logTime = 1/logHz;//s interval between logging data

    //Pressurise the actuator under test
    leg.StartLogging();
    leg.setValve(true);
    
    //until the user presses the key to stop datalogging or the target max test
    //pressure is reached
    while(leg.isLogging() && (leg.getPressure0()*100 < targetkPa)) {        
        leg.pc.printf("\r\nPressure (kPa): \t%7.2f",leg.getPressure0()*100);//print the pressure
        leg.LogData();//record a datapoint
        wait(logTime);//wait
    }
    
    //Depressurise when the user is done with benchmarking or the target pressure is
    //reached    
    leg.pc.printf("\r\nExit Failure Test");
    leg.setValve(false);
    leg.StopLogging(); //Stop logging data
    leg.resumePrint(); //Let the Bench class print    
}

/**
 * A method used to benchmark the actuator under test before the linear actuator
 * was fitted to the test rig. It simply pressurises the actuator and records data
 * @param logHz: datalogging frequency
 */
void runBenchmarkExperiment0(int logHz)
{
    double logTime = 1/logHz;//s interval between logging data
    
    //Pressurise the actuator under test
    leg.StartLogging();
    leg.setValve(true);//inflation
    while(leg.isLogging()) {//until the user presses the key to stop datalogging        
        leg.pc.printf("\r\nPressure (kPa): \t%7.2f",leg.getPressure0()*100); //print the pressure
        leg.LogData();//record a datapoint
        wait(logTime);//wait
    }
    
    //Depressurise when the user is done with benchmarking   
    leg.pc.printf("\r\nExit Benchmarking Test"); 
    leg.setValve(false);//deflate
    leg.StopLogging(); //Stop logging data
    leg.resumePrint(); //Let the Bench class print    
}

/**
 * A benchmark test with the linear actuator. The actuator is cycled from its flexed position,
 * to the point at which it no longer produces force, and back again. This is considered its range of motion.
 * NOTE - not extensively tested
 * @param minForce: (N) force at which cycle goes from extension to flexion
 * @param maxAngle: (deg) angle at which cycle goes from flexion to extension
 * @param inflationT: (s) time allowed for actuator to inflate
 * @param pwm: actuator speed (in %, from 0-100. > 50 is needed to move actuator in practice).
 * @param cycles: the number of up and down cycles of the leg
 * @param logHz: datalogging frequency
 */
void runBenchmarkExperiment1(int minForce, float maxAngle, float inflationT, int pwm, int cycles, int logHz)
{
    double logTime = 1/logHz;//s interval between logging data
    
    //Stop the Bench class from printing, so this method can print
    leg.pausePrint();

    //Pressurise the actuator, giving it inflationT to do so
    leg.setValve(true);
    Timer t;
    t.reset();
    t.start();
    while(leg.isLogging() && t.read() < inflationT) {
        leg.pc.printf("\r\nPressure (kPa): \t%7.2f",leg.getPressure0()*100);//print the pressure
        wait(0.2);//wait a bit
    }
    
    //Now actuator is inflated, the test can start
    leg.StartLogging();
    // in loop
    for (int i=0; i<cycles; i++) {
        //allow leg to go up until force is below threshold
        leg.setPWM(pwm);
        leg.setDir(0);
        
        leg.pc.printf("\r\nExtending: \t%i",cycles);//print which cycle we are on
        
        //continue while force is above threshold and the user has not stopped logging
        while((leg.getForce() >= minForce) && leg.isLogging()) {
            leg.pc.printf("\r\nForce (N): \t%7.2f",leg.getForce());//feedback on force printed
            leg.LogData();//record datapoint
            wait(logTime);//wait
        }
        
        //If the user stopped
        if (!leg.isLogging()) {
            leg.pc.printf("\r\nUser stopped during extension Exit");
            leg.setPWM(0); //turn off linear actuator
            leg.setValve(false);//Depressurise
            leg.StopLogging(); //Stop logging data
            leg.resumePrint(); //Let the Bench class print
            return;
        }

        //allow leg to go down until angle is greater than threshold
        leg.setPWM(pwm);
        leg.setDir(1);
        
        leg.pc.printf("\r\nFlexing: \t%i",cycles); //print which cycle we are on
        
        //continue while the angle is less than the threshold and the user has not stopped logging
        while((leg.getDegrees(0) <= maxAngle) && leg.isLogging()) {
            leg.pc.printf("\r\nAngle (deg): \t%7.2f",leg.getDegrees(0));//feedback on angle printed
            leg.LogData();//record datapoint
            wait(logTime);//wait
        }
        
        //If the user stopped
        if (!leg.isLogging()) {
            leg.pc.printf("\r\nUser stopped during flexion Exit");
            leg.setPWM(0); 
            leg.setValve(false);//Depressurise
            leg.StopLogging(); //Stop logging data
            leg.resumePrint(); //Let the Bench class print
            return;
        }
    }
    
    //Depressurise when the desired number of cycles are completed
    leg.pc.printf("\r\nExit Benchmarking Test");
    leg.setPWM(0);//motor off
    leg.setValve(false);//actuator depressurised
    leg.StopLogging(); //Stop logging data
    leg.resumePrint(); //Let the Bench class print
}

