/*  Copyright 2020 Allan Joshua Veale
 
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
 
        http://www.apache.org/licenses/LICENSE-2.0
 
    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#ifndef _BENCH02_H_
#define _BENCH02_H_
#include "MODSERIAL.h"

#include "mbed.h"
#include <string>

#include "SDFileSystem.h"
#include "constants.h"

#include "as5048.h"
#include "lcm101.h"
#include "SPTEPressureSensor.h"
#include "Valve.h"
#include "LinearActuator.h"

/**
 * Class to read out sensory information from the test rig:
 * the knee joint angle as measured by an AMS AS5048 absolute rotary sensor
 * the vertical force applied externally at the hip joint
 * the pressure (for example of an actuator used in the device under test).
 * Additionally, an on /off valve can pressurise the actuator on the leg and a linear
 * actuator can be used to raise and lower the test leg
 */
class Bench {
public:
    
    //angle is only allowed to be< this value 
    static const float kCutOffDegrees = 180.0f; 
    static const float kCutOffRadians = 3.14159265359f;
   
   
    // constructor
    Bench(PinName mosi = AS5048_MOSI, PinName miso = AS5048_MISO, PinName sck = AS5048_SCLK, PinName cs = AS5048_CS, 
        bool use5kN = sensors::use5kN,PinName p_lcm101 = LCM101, 
        PinName p_spte0 = SPTE_0, PinName p_spte1 = SPTE_1,PinName p_valve = VALVE_PIN,PinName p_laDir = LINACT_DIR_PIN, PinName p_laPwm = LINACT_PWM_PIN,
        PinName sd_mosi = SD_MOSI, PinName sd_miso = SD_MISO, PinName sd_sck = SD_SCK, PinName sd_cs = SD_CS,
        PinName tx = USBTX, PinName rx = USBRX);  

    
    //setup
    void initialise();
    
    void setLoggingFrequency(float logHertz); //set datalogging frequency (for when automatic data logging is used - not in this code)
    void setExtraColumns(string extraColumnNames[], int numCols);//manually add extra columns of data so user can have custom data recorded
    void setExtraData(float data[]);//manually add extra datapoints
    void update(); //read angle data afresh

    bool isLogging();//indicates if we are now datalogging
    void stopLogging();//halts the manual data logging if we are now datalogging
    void StopLogging(); //halts the automatic data logging if we are now datalogging
    void StartLogging(const char * fname_append = "data"); //starts datalogging (now manual)
    void RestartLogging(const char * fname_append = "data");//restart datalogging
    void LogData(); //records the predetermined datapoints (and those in extracolumns)
    
    //for reading angle sensors
    As5048* get_as5048();
    
    float getDegrees(int i_joint); //read angle
    const char* getJointName(int i_joint);//return joint name

    //for reading and calibrating force and pressure sensors
    float getForce();//read force
    void nullForce();//tare reading
    
    float getPressure0();//read pressure
    void nullPressure0();//tare reading
    float getPressure1();//read pressure
    void nullPressure1();//tare reading
    
    
    //for controlling valve
    bool getValve();//return valve state
    void setValve(bool pressurise); //turn valve on/off
    
    // controlling linear actuator
    void setDir(bool dir);//set actuator direction
    bool getDir();//return actuator direction
    void setPWM(int pwm);  //actuator speed  
    
    /** Object for printing/receiving data to the PC the test rig is connected 
    * to. Note that it uses the MODSERIAL.h class.
    */
    MODSERIAL pc;
    
    void pausePrint();//stops automatic printing of data
    void resumePrint();//resumes automatic printing of data
    
private:
    //timing
    int loggingUS; //microseconds between recording data logging
    float loggingHz;// Hz of data logging
    
    Ticker tick_update, tick_serial, tick_logging; //for the updating of sensor values, printing of serial data and logging of data
    Timer timer;
    Timer keyIntT;//timer to debounce keypresses
    int keyIntOffset;//for setting initial value of keyIntT
    int firstReadMS; //first timer value read
    bool startedLogging; //in the middle of a logging cycle
    
    bool is_logging;
    bool is_printing;
    bool was_printing; //if printing is paused true, or if is_printing was true
    char k; //value from keyboard press
            
    //joint angle 
    As5048 as5048_;
    
    //load cells
    Lcm101 loadCell5kN;
    Lcm101 loadCell1kN;
    bool use5kN;//determine which load cell is used
    
    //pressure sensors
    SPTEPressureSensor spte0;
    SPTEPressureSensor spte1;
    
    //valve
    ValveDigital valveFesto;
    
    //linear actuator
    LinearActuator LinAct;
    
    //SD card
    SDFileSystem sd;
    FILE * fp_data;
    int fname_prepend;//file number
    bool sd_card_present;//card detected    
    void InitSdCard(); //setup the SD card
    
    // recording extra data   
    static const int maxCols = 5;
    int usedExtraCols; //number of extra columns useds
    string extraColNames[maxCols];//names of extra columns (up to maxCols allowed with 15 characters)  
    float extraColValues[maxCols];//value held in extra columns for current time step
                

    //for toggling logging or printing
    void ToggleState(MODSERIAL_IRQ_INFO *q);//method for parsing keyboard presses

    void ToggleLogging();//change state of logging (between on and off)
    
    // serial printing
    void TogglePrinting(); //change state of printing (between on and off)
    void PrintStatus();//print the read out from the test rig sensors
    void PrintMenu();  //print the standard menu showing which keys control what in the test rig
};
#endif