#include "mbed.h"
#include "XMIT_IR.h"


IRTransmitter::IRTransmitter(PinName TX, PinName PWM, PinName RX, int data_buff_len, void (*fptr_isr_rx)(void))
{
    ir_serial = new Serial(TX,RX);
    ir_pwm = new PwmOut(PWM);
    packet_buffer_out = new char[data_buff_len]();
    packet_buffer_in = new char[data_buff_len+3]();

    //Initialize modules
    (*ir_pwm).period(1.0/38000.0);  //38kHz Modulation Freq
    (*ir_pwm) = 0.5;                //pulse width = 50%
    (*ir_serial).baud(2400);
    (*ir_serial).attach(fptr_isr_rx,Serial::RxIrq);

    //Initialize Variables
    for(int i=0; i<data_buff_len+3; i++)
        packet_buffer_in[i] = 0;
}

IRTransmitter::~IRTransmitter()
{
    delete ir_serial;
    delete ir_pwm;
    delete packet_buffer_out;
    delete packet_buffer_in;
}

void IRTransmitter::set_debug_port(Serial* port)
{
    debug_port = port;
}


void IRTransmitter::MakePacket(char* data,int len)
{
    char check =0x0;
    
    (*ir_serial).putc(ASCII_STX);
    for(int i=0; i<len; i++) {
        check^=data[i];
        (*ir_serial).putc(data[i]);
    }
    (*ir_serial).putc(check);
    (*ir_serial).putc(ASCII_ETX);

}



char IRTransmitter::ReadPacket(char* data, int len)
{
    //Run this every few milliseconds to run through mailbox and check packet data.
    //The data array will update with new values once a packet has been validated.
    //Returns true if new data, false otherwise
    //Inputs:
    //          -data: a character array to store data variables
    //          -len:  length of character array

    char return_state=0;
    osEvent evt = ir_rxdata_mailbox.get();
    while(evt.status == osEventMail) {
        char* mail = (char*)evt.value.p;
        (*data) = (*mail);

        //check for valid packet - update data if so
        update_packet_buffer(*data,len);

        /*(*debug_port).printf("\n");
        for(int i=0; i<len+3; i++) {
            (*debug_port).printf("0x%02X.",packet_buffer_in[i]);
        }
        (*debug_port).printf("  PKT_VALID=0x%02X.",packet_is_valid(len));*/



        if(packet_is_valid(len)==0) {
            return_state = 1;
            for(int i=0; i<len; i++) {
                data[i] = packet_buffer_in[i+1];
            }
        }

        ir_rxdata_mailbox.free(mail);
        if(return_state) {
            return return_state;
        }

        evt = ir_rxdata_mailbox.get();
    }
    return return_state;
}

char IRTransmitter::packet_is_valid(int data_len)
{

    char check = 0;
    //check for valid packet
    if(packet_buffer_in[0]!=ASCII_STX)
        return 0x1; //bad start byte
    if(packet_buffer_in[data_len+2]!=ASCII_ETX)
        return 0x2; //bad end byte

    for(int i=1; i<data_len+1; i++) {
        check^=packet_buffer_in[i];
    }
    if(check!=packet_buffer_in[data_len+1]) {
        return 0x3; //bad checksum
    }

    return 0;
}

void IRTransmitter::update_packet_buffer(char new_data, int data_len)
{

    //Shift All data 1 cell over
    for(int i=0; i<data_len+2; i++) {
        packet_buffer_in[i] = packet_buffer_in[i+1];
    }
    packet_buffer_in[data_len+2] = new_data;
}


void IRTransmitter::isr_rx(char data)
{
    char* mail = ir_rxdata_mailbox.alloc();
    mail[0] = (char)data;
    ir_rxdata_mailbox.put(mail);
}


