#include "mbed.h"
#include "rtos.h"
#include "NokiaLCD.h"
#include "XMIT_IR.h"

#define FPS 15
#define PKT_RATE 15

/****************************************
|=======================================|
|MBED Connections:                      |
|   -p5 : DIO on Sparkfun Nokia LCD     |
|   -p7 : CLK on Sparkfun Nokia LCD     |
|   -p8 : CS  on Sparkfun Nokia LCD     |
|   -p9 : RST on Sparkfun Nokia LCD     |
|   -p21: CTL on Sparkfun IR Xmitter    |
|   -p14: OUT on Sparkfun IR Rcvr       |
|   -p13: GND on Sparkfun IR Xmitter    |
|=======================================|
****************************************/

//Function Prototypes
void BlinkAlive(void const* arguments);
void UpdateLCD(void const* arguments);
void IRStuff(void const* arguments);
void ISR_UARTRX(void);
char CheckPacket(char new_data, char* packet_buffer, char* data, int data_len);
void MakePacket2(char* data,int len);

//Pin Setup
PwmOut led1(LED1);
PwmOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
NokiaLCD lcd(p5, p7, p8, p9, NokiaLCD::LCD6610); // mosi, sclk, cs, rst, type
Serial* devicerx;

//Serial devicetx(p13,NC);
Serial pc(USBTX,USBRX);
PwmOut IRLED_mod(p21);

IRTransmitter ir_tx(p13,p21,p14,4,&ISR_UARTRX); //tx,pwm,size

//Global Vars
char text_buffer[32];
char irdatOUT[10];
char irdatIN[10];
char irdata_out=0;
char error_code=0;
Thread* threadptr_irstuff;
char packet_buff[10];
Mutex data_update_mutex;




int main()
{

    //LCD init
    lcd.background(0x000000);

    //PC serial init
    pc.baud(19200);
    pc.printf("Starting...\n\n");

    //IR_TRX module
    ir_tx.set_debug_port(&pc);


    //Variable Init
    for(int i=0; i<10; i++) packet_buff[i]=0;

    //Threads init
    Thread thread_blinkalive(BlinkAlive);
    Thread thread_updatelcd(UpdateLCD);
    Thread thread_irstuff(IRStuff);
    threadptr_irstuff = &thread_irstuff;



    while(1) {

        //Use main loop to set LCD framerate
        thread_updatelcd.signal_set(0x1);
        thread_blinkalive.signal_set(0x1);
        Thread::wait(1000/FPS);

    }
}

void UpdateLCD(void const* arguments)
{
    while(true) {

        //Start flash LED
        led2 = 0.02;
        data_update_mutex.lock();
        char irdatbuff[4];
        for(int i=0; i<4; i++) irdatbuff[i] = irdatIN[i];
        data_update_mutex.unlock();

        if(irdatbuff[0]!=0xA5 && irdatbuff[0]!=0x00) {
            lcd.locate(0,7);
            lcd.printf("Big Problem!  0x%02X",irdatbuff[0]);
            while(1);
        }


        //Write debug text to screen
        lcd.locate(0,1);
        lcd.printf("Debug:");
        lcd.locate(0,3);
        time_t seconds = time(NULL);
        strftime(text_buffer, 32, "%I:%M:%S %p\n", localtime(&seconds));
        lcd.printf("%s", text_buffer);
        lcd.locate(0,4);
        lcd.printf("IR_OUT=0x%02X,0x%02X", irdatOUT[0],irdatOUT[1]);
        lcd.locate(0,5);
        lcd.printf("IR_IN= 0x%02X,0x%02X", irdatbuff[0],irdatbuff[1]);
        lcd.locate(0,6);
        lcd.printf("Error= 0x%02X", error_code);

        //End - flash LED
        led2 = 0.0;
        //End - Sleep thread
        Thread::signal_wait(0x1);
    }
}

void IRStuff(void const* arguments)
{
    while(true) {

        //If data available - Print data directly to USB port (for debug)

        char result;
        char temp_buff[4];
        
        //buffer data into temp register
        result = ir_tx.ReadPacket(temp_buff,4);

        if(result == 0x1) {
            pc.printf("\n");
            //update data - mutex
            data_update_mutex.lock();
            for(int i=0; i<4; i++) {
                irdatIN[i] = temp_buff[i];
            }
            data_update_mutex.unlock();
            for(int i=0; i<4; i++) {
                pc.printf("0x%02X.",irdatIN[i]);
            }
        }



        //Do not return until we have more data
        //Thread::signal_wait(0x1);
        Thread::wait(10);
    }

}



void BlinkAlive(void const* arguments)
{
    while(true) {

        //Change LED1 state (debug)
        (led1==0.0)?led1=0.02:led1=0.0;

        //Form a test packet and send it over IR transmitter
        irdatOUT[0] = 0xA5;
        irdatOUT[1] = ++irdata_out;
        irdatOUT[2] = ~(irdata_out*2);
        irdatOUT[3] = irdata_out*7+13;
        ir_tx.MakePacket(irdatOUT,4);

        //Roughly use to set rate of data packets per second
        //Thread::wait(100);
        Thread::signal_wait(0x1);
    }
}




//Handle Reception of RX data (mail it to appropriate thread)
void ISR_UARTRX()
{
    //get RX data (and prevent ISR from looping forever
    uint32_t RBR = LPC_UART1->RBR;
    ir_tx.isr_rx(RBR);

}



