#ifndef EASYVR_H_DEFINED
#define EASYVR_H_DEFINED

#include <mbed.h>

#define ARG_MIN  '@'
#define ARG_MAX  '`'
#define ARG_ZERO 'A'
#define ARG_ACK  ' '

#define CMD_BREAK      'b'
#define CMD_SLEEP      's'
#define CMD_KNOB       'k'
#define CMD_LEVEL      'v'
#define CMD_LANGUAGE   'l'
#define CMD_TIMEOUT    'o'
#define CMD_RECOG_SI   'i'
#define CMD_TRAIN_SD   't'
#define CMD_GROUP_SD   'g'
#define CMD_UNGROUP_SD 'u'
#define CMD_RECOG_SD   'd'
#define CMD_ERASE_SD   'e'
#define CMD_NAME_SD    'n'
#define CMD_COUNT_SD   'c'
#define CMD_DUMP_SD    'p'
#define CMD_MASK_SD    'm'
#define CMD_RESETALL   'r'
#define CMD_ID         'x'
#define CMD_DELAY      'y'
#define CMD_BAUDRATE   'a'
#define CMD_QUERY_IO   'q'
#define CMD_PLAY_SX    'w'
#define CMD_DUMP_SX    'h'

#define STS_MASK       'k'
#define STS_COUNT      'c'
#define STS_AWAKEN     'w'
#define STS_DATA       'd'
#define STS_ERROR      'e'
#define STS_INVALID    'v'
#define STS_TIMEOUT    't'
#define STS_INTERR     'i'
#define STS_SUCCESS    'o'
#define STS_RESULT     'r'
#define STS_SIMILAR    's'
#define STS_OUT_OF_MEM 'm'
#define STS_ID         'x'
#define STS_PIN        'p'
#define STS_TABLE_SX   'd'

#define ERR_DATACOL_TOO_NOISY  0x3
#define ERR_DATACOL_TOO_SOFT   0x4
#define ERR_DATACOL_TOO_LOUD   0x5
#define ERR_DATACOL_TOO_SOON   0x6
#define ERR_DATACOL_TOO_CHOPPY 0x7
#define ERR_RECOG_FAIL         0x11
#define ERR_RECOG_LOW_CONF     0x12
#define ERR_RECOG_MID_CONF     0x13
#define ERR_RECOG_BAD_TEMPLATE 0x14
#define ERR_RECOG_DURATION     0x17
#define ERR_SYNTH_BAD_VERSION  0x4A
#define ERR_SYNTH_BAD_MSG      0x4E
#define ERR_NOT_A_WORD         0x80

#define i2a(i) ((char)((i)+ARG_ZERO))
#define a2i(a) ((int)((a)-ARG_ZERO))

extern Serial term;

/** EasyVR class for ECE 4180 lab 4.
 *  Used for interfacing with an EasyVR module.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "easyvr.h"
 *
 * EasyVR easyVR( p27, p28 );
 * 
 * int main()
 * {
 *     easyVR.wakeup();
 *     easyVR.setup();
 * }
 * @endcode
 */
class EasyVR
{
    public:
        /** EasyVR constructor.
         *  Initializes the serial interface for the EasyVR module.
         *
         * @param tx The TX pin on the mbed that should be used. This connects to the EasyVR's RX pin.
         * @param rx The RX pin on the mbed that should be used. This connects to the EasyVR's TX pin.
         */
        EasyVR( PinName tx, PinName rx );
        
        /** Wakes up the EasyVR by sending it a "break" signal until a reply
         *  of "success" has been received.
         */
        void wakeup();
        
        /** Sets up the EasyVR. This includes ensuring the EasyVR module is a valid EasyVR device,
         *  setting its timeout value, and its language.
         *
         * @param lang The language to be used, as an encoded integer. i.e. 0 is 'A', 1 is 'B', and so forth.
         * @param timeout The timeout to be used, in seconds, as an encoded integer.
         */
        void setup( int lang, int timeout );
        
        /** Sends a single byte to the EasyVR. This is most commonly used to send commands
         *  and parameters for those commands. Note that this is just a wrapper for the EasyVR's
         *  serial interface's putc() function.
         *
         * @param byte The byte to be sent.
         * @see Serial.putc()
         */
        void send( char byte );
        
        /** Receives a single byte from the EasyVR. This is most commonly used to receive
         *  signals from the EasyVR. Note that this is just a wrapper for the EasyVR's serial
         *  interface's getc() function.
         *
         *  Be aware that this is a blocking method - it will stall the program until a byte is received!
         *
         * @return A single byte from the EasyVR.
         * @see Serial.putc()
         */
        char receive();
        
        /** Sets the EasyVR's baud rate. For speech recognition, this must be exactly 9600.
         *
         * @param rate The baud rate the EasyVR should communicate with.
         * @see Serial.baud()
         */
        void baud( int rate );
    
    private:
        Serial dev;
};

#endif // EASYVR_H_DEFINED