/* WizFi310 implementation of NetworkInterfaceAPI
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 

#include "WizFi310Interface.h"

// Various timeouts for different WizFi310 operations
#define WizFi310_CONNECT_TIMEOUT 15000
#define WizFi310_SEND_TIMEOUT    500
#define WizFi310_RECV_TIMEOUT    0
#define WizFi310_MISC_TIMEOUT    500

#define WizFi310_DELAY_MS       300


// WizFi310Interface implementation
WizFi310Interface::WizFi310Interface(PinName tx, PinName rx, PinName cts, PinName rts, PinName reset, PinName alarm,  int baud)
    : _wizfi310(tx, rx, cts, rts, reset, alarm, baud)
{
    memset(_ids, 0, sizeof(_ids));
}

int WizFi310Interface::connect(
    const char *ssid,
    const char *pass,
    nsapi_security_t security)
{
    if (!_wizfi310.startup()) 
    {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    _wizfi310.setSsid(ssid);
    _wizfi310.setSec(security, pass);
    _wizfi310.setAddress("");
    
    if( _wizfi310.join(WizFi310::WM_STATION) == -1)
    {
        return NSAPI_ERROR_NO_CONNECTION;
    }

    return 0;
}

int WizFi310Interface::disconnect()
{
    if ( _wizfi310.cmdWLEAVE() == -1 )  return NSAPI_ERROR_DEVICE_ERROR;

    return 0;
}

const char *WizFi310Interface::get_ip_address()
{
    return _wizfi310.getIPAddress();
}

const char *WizFi310Interface::get_mac_address()
{
    return _wizfi310.getMACAddress();
}

int WizFi310Interface::conTP(
    const char *clientId, 
    const char *credentialId, 
    const char *serviceId, 
    const char *devId, 
    const char *containerNm,
    const char *commandName
    )
{
    //if ( _wizfi310.joinTP(clientId, credentialId, serviceId, devId, containerNm) == -1 ) return NSAPI_ERROR_NO_SOCKET;
    
    if(_wizfi310.cmdSKTPCON("1", clientId, credentialId, serviceId, devId))    return -1;
    WIZ_INFO("ThingPlug Connected");
    
    if(_wizfi310.cmdSKTPDEVICE("1", devId))    return -1;    
    WIZ_INFO("Device Registered");
    
    if(_wizfi310.cmdSKTPCONTAINER("1", containerNm))    return -1;
    WIZ_INFO("Created Container\r\n");
    
    if(_wizfi310.cmdSKTPCMD("1", commandName))    return -1;
    WIZ_INFO("Created CommandName\r\n");
    
    return 0;
}

int WizFi310Interface::sendTP(const char *containerNm, const char *sendData)
{
    if(_wizfi310.cmdSKTPSEND(containerNm, sendData))    return -1;
    WIZ_INFO("Data sent\r\n");
    
    return 0;
}

int WizFi310Interface::recvTP(const char *commandName, int executeStatus, int executeResult)
{
    int cnt, cid = 0, ret = 0;
    char buffer[1024] = "";
       
    if(_wizfi310.recv(cid, buffer, sizeof(buffer)))
    {
        //printf("%s\r\n", buffer);
        
        if(_wizfi310.cmdSKTPRESULT(commandName, executeStatus, executeResult))  
        {
            _wizfi310.initCon(cid, true);
            
            return -1;
        }
                
        for(cnt = 0; cnt < sizeof(buffer); cnt++)
        {
            if( buffer[cnt] == '8' )
            {
                if( buffer[cnt + 1] == '0' )
                    ret = 1;
                    
                else if( buffer[cnt + 1] == '1' )
                    ret = 2;
            }
        }
        
        WIZ_INFO("Complete");
        _wizfi310.initCon(cid, true);
        
        return ret;
    }
    
    //WIZ_INFO("Incorrect Request\r\n");
    _wizfi310.initCon(cid, true);
    
    return -1;
}

int WizFi310Interface::disConTP()
{   
    if(_wizfi310.cmdSKTPCON("0"))    return -1;
    WIZ_INFO("ThingPlug Disconnected\r\n");
    
    return 0;
}

struct wizfi310_socket {
    int id;
    nsapi_protocol_t proto;
    bool connected;
};

int WizFi310Interface::socket_open(void **handle, nsapi_protocol_t proto)
{
    // Look for an unused socket

    /*
    int id = -1;
 
    for (int i = 0; i < WIZFI310_SOCKET_COUNT; i++) {
        if (_ids[i] == false) {
            id = i;
            _ids[i] = true;
            break;
        }
    }
 
    if (id == -1) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    */
    
    struct wizfi310_socket *socket = new struct wizfi310_socket;
    if (!socket) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    socket->id = -1;
    socket->proto = proto;
    socket->connected = false;
    *handle = socket;
    return 0;
}

int WizFi310Interface::socket_close(void *handle)
{
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;
    int err = 0;
 
    if(socket->id == -1){
        err = NSAPI_ERROR_NO_SOCKET;
    }
    else if (_wizfi310.close(socket->id) == -1) {
        err = NSAPI_ERROR_DEVICE_ERROR;
    }

    _ids[socket->id] = false;
    wait_ms(WizFi310_DELAY_MS);
    delete socket;
    return err;
}

int WizFi310Interface::socket_bind(void *handle, const SocketAddress &address)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi310Interface::socket_listen(void *handle, int backlog)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi310Interface::socket_connect(void *handle, const SocketAddress &addr)
{
    int cid=-1;
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;

    WizFi310::Protocol proto = (socket->proto == NSAPI_UDP) ? WizFi310::PROTO_UDP : WizFi310::PROTO_TCP;
    if((cid = _wizfi310.open(proto, addr.get_ip_address(), addr.get_port())) == -1 )
    {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    if(cid >= WIZFI310_SOCKET_COUNT)
    {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    _ids[cid] = true;
    socket->id = cid;
    socket->connected = true;
    wait_ms(WizFi310_DELAY_MS);
    return 0;
}
    
int WizFi310Interface::socket_accept(void **handle, void *server)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi310Interface::socket_send(void *handle, const void *data, unsigned size)
{
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;
 
    if ( _wizfi310.send(socket->id, (const char*)data, size) == -1 ) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }
 
    return size;
}

int WizFi310Interface::socket_recv(void *handle, void *data, unsigned size)
{
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;
 
    int32_t recv = _wizfi310.recv(socket->id, (char*)data, size);
    if (recv <= 0) {
        return NSAPI_ERROR_WOULD_BLOCK;
    }
 
    return recv;
}

int WizFi310Interface::socket_sendto(void *handle, const SocketAddress &addr, const void *data, unsigned size)
{
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;
    if (!socket->connected) {
        int err = socket_connect(socket, addr);
        if (err < 0) {
            return err;
        }
    }
    
    return socket_send(socket, data, size);
}

int WizFi310Interface::socket_recvfrom(void *handle, SocketAddress *addr, void *data, unsigned size)
{
    struct wizfi310_socket *socket = (struct wizfi310_socket *)handle;    
    return socket_recv(socket, data, size);
}

void WizFi310Interface::socket_attach(void *handle, void (*callback)(void *), void *data)
{
}
