#include "mbed.h"
#include <limits.h>
//
#include "XMONlib.h"
//
#include "vawe_data.h"
#include "sine_220.h"
#include "sine_440.h"
#include "sine_880.h"

//
DigitalOut myled(LED1);
Timer t;

/** Utility function to plot or view data to XMON.
 */
void plot_spectrum( void);
void plot_sgnl_src( void);
void plot_sgnl_out( void);
void view_fft_timeval( void);
void view_spectrum_timeval( void);

//
XMON xm( USBTX, USBRX);

/** Data structure about fft, ifft and spectrum calculation time
 *  The  FFT process is a sum of different operation. More ofthen we need only the
 *  FFT+Spectrum calculation. Sometime can be useful to FFT and then iFFT the sample
 *  after some change. So the exact time consuming operation is important to know.
 */
struct _timemon {
    float fft_tm;
    float ifft_tm;
    float cpy_tm;
    float spec_tm;
    int   maxbin_idx;
    int   maxbin_val;
    float freq;
};    

#define NBIN    1024
int spectrum[NBIN/2];
short buff[NBIN];
//
short x[NBIN*2]; // input data 16 bit, 4 byte aligned  x0r,x0i,x1r,x1i,....
short y[NBIN*2]; // output data 16 bit,4 byte aligned  y0r,y0i,y1r,y1i,....
short z[NBIN*2]; // same format...

// *****************************************************************
// code from FFTCM3.s by Ivan Mellen
// http://www.luminarymicro.com/component/option,com_joomlaboard/Itemid,92/func,view/id,1636/catid,6/
extern "C" void fftR4(short *y, short *x, int N);
extern "C" void ifftR4(short *y, short *x, int N);
// *****************************************************************

//
struct _timemon time_mon;

/** Execute the FFT and iFFT.
 * 
 * @param vd a pointer to the VAWE_DATA structure. See vawe_data.h for the declaration.
 */
void doFFT( VAWE_DATA *vd);

void doFFT( VAWE_DATA *vd)
{
    // zeroed the x array...
    for (int i=0;i<(NBIN*2);i++) x[i]=0;
    
    t.reset();
    // start the timer to count copy value.
    t.start();
    for (int i=0; i<NBIN*2; i+=2) {
        //
        x[i] = vd->pdata[i/2];
    }
    t.stop();
    time_mon.cpy_tm=t.read();        
    t.reset();
        
    // the FFT function...
    t.start();
    fftR4(y, x, NBIN);   // y is in frequency domain 
    t.stop();
    time_mon.fft_tm=t.read();
    t.reset();
    
    // ... do something with the signal...
    
    t.start();
    ifftR4(z, y, NBIN);  // z should be x/N + noise introduced by 16 bit truncating 
    t.stop();
    time_mon.ifft_tm=t.read();
    t.reset();

    // calculating spectrum...
    time_mon.maxbin_val=time_mon.maxbin_idx=0;
    t.start();
    
    /* spectrum[i]=sqrt( real[i]^2 + img[i]^2) */
    for (int i=0; i<NBIN; i+=2)
    {
        spectrum[i/2] = (int)sqrt( float(y[i]*y[i] + y[i+1]*y[i+1]) );
        // calculate and save the max bin value and index...
        if ( time_mon.maxbin_val < spectrum[i/2]) {
            time_mon.maxbin_val = spectrum[i/2];
            time_mon.maxbin_idx = i/2;
        }
    }
    t.stop();
    time_mon.spec_tm=t.read();
    // Calculate the freq. of the higher spectrum signal.
    // freq=Max_Bin*(Sampling_Rate/Num_Sample);
    time_mon.freq=(float)((float)time_mon.maxbin_idx*(float)( (float)vd->srate/(float)vd->samples));    
}


int main()
{
    unsigned char k;
    
    // Set the rate to XMON. Remenber to change the program too...
    xm.baud( 115200);
    
    // Generate the first FFT and iFFT value...
    doFFT( &sine_220);
    // Plot and view to XMON...
    plot_spectrum();
    view_spectrum_timeval();
 
    // Loop waiting some character from the XMON programm   
    while( 1) {
        //
        while( !xm.readable());
        //
        k=xm.getc();
        // Plot the spectrum. [s]
        if ( k=='s') {
            plot_spectrum();
        }
        // Plot the source signal [i]
        if ( k=='i') {
            plot_sgnl_src();
        }
        // Plot the signal after the iFFT [o]
        if ( k=='o') {
            plot_sgnl_out();
        }
        // View the time values about FFT process [f]
        // 1. time to copy  value
        // 2. time to calculate the FFT
        // 3. time to calculate the spectrum
        // 4. time to calculate ti iFFT
        if ( k=='f') {
            view_fft_timeval();
        }        
        // View the time values about spectrum and freq calculation of max bin [b]
        // 1. Energy of the max bin.
        // 2. The max bin.
        // 3. Freq. calcualted from the max bin.
        if ( k=='b') {
            view_spectrum_timeval();
        }                
        // Make calculation on freq 1
        if ( k=='1') {
            doFFT( &sine_220);
            plot_spectrum();
            view_spectrum_timeval();
        }
        // Make calculation on freq 2
        if ( k=='2') {
            doFFT( &sine_440);
            plot_spectrum();
            view_spectrum_timeval();
        }
        // Make calculation on freq 3
        if ( k=='3') {
            doFFT( &sine_880);
            plot_spectrum();
            view_spectrum_timeval();
        }
    }
        
    while( 1);
    
}

void plot_spectrum( void)
{
    xm.plotZero();
    for (int i=0; i<NBIN/2; i++)
    {
        xm.plot( spectrum[i], XMON_RED);
    }

}

void plot_sgnl_src( void)
{
    xm.plotZero();
    for (int i=0; i<NBIN; i+=2)         // display only half data
    {
        xm.plot( x[i], XMON_GREEN);
    }

}

void plot_sgnl_out( void)
{
    xm.plotZero();
    for (int i=0; i<NBIN; i+=2)         // display only half data
    {
        xm.plot( z[i], XMON_BLUE);
    }

}

void view_fft_timeval( void)
{
    xm.viewf( time_mon.cpy_tm, 0);      // copy data from a real array to real+img...
    xm.viewf( time_mon.fft_tm, 1);      // FFT calculation...   
    xm.viewf( time_mon.spec_tm, 2);     // spectrum calculation...
    xm.viewf( time_mon.ifft_tm, 3);     // iFFT calculation...
}

void view_spectrum_timeval( void)
{
    xm.view( time_mon.maxbin_val, 0);   // The energy value of the max bin.
    xm.view( time_mon.maxbin_idx, 1);   // The max bin.
    xm.viewf( time_mon.freq, 2);        // The freq. calculated from the max bin.
    xm.view( 0, 3);

}