    /* mbed Graphics LCD library. Library for MI0283QT-2 screen.

    Copyright (c) 2011 NXP 3803 
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef __MBED_GLCDLIB_H
#define __MBED_GLCDLIB_H

// Comment out if file system functions are not used.
#define _USE_FILE

#ifdef _USE_FILE
#define BUFFER_LINE     10
#endif

//
#define LCD_BLACK   (0)
#define LCD_WHITE   ( (255<<16)|(255<<8)|(255<<0) )
#define LCD_RED     ( (255<<16)|(0<<8)|(0<<0) )
#define LCD_GREEN   ( (0<<16)|(255<<8)|(0<<0) )
#define LCD_BLUE    ( (0<<16)|(0<<8)|(255<<0) )
//
#define SCALE_320x240    1
#define SCALE_160x120    2
//
#include "mbed.h"

struct _FONTINFO {
    unsigned char *pText;
    unsigned int h_size;
    unsigned int v_size;
    unsigned int h_line;
};


/** Graphic LCD Library for MI0283QT-2 LCD
* 
* Image Format.
* About image format, I choose the way to make the library capable to read the simple RGB format.
* That's: a file with RGB bytes written. This format is very simple to obtain, using the convert.exe software.
* This program is inside the "Imagemagick" installation software. You can download the portable versio for windows 
* at this link: http://www.imagemagick.org/download/binaries/ImageMagick-6.6.7-Q16-windows.zip
*
* Icone Format.
* I use some free icons from Internet. I use the bin2h utility to convert this file to .h format.
* (http://www.deadnode.org/sw/bin2h/)
* 
* Example:
* @code
* // Init code...
* #include "mbed.h"
* #include "GLCDlib.h"
* 
* // Configure the GLCD pin: 
* //         mosi, miso, clk, cs,  rst, bklgh
* GLCD lcd(  p11,  p12,  p13, p14, p17, p26);
* 
* int main() {
*
*    lcd.lcd_init();
*    lcd.backlightoff();
*    //... do somethings...
*   lcd.lcd_clear( LCD_WHITE);
*    lcd.backlighton();
*    //... the LCD is ON
* }
* @endcode

*/
    
class GLCD {

public:
    /** Create the GLCD Object and initlaize the I/O pins
     * 
     * @param pin mosi, pin miso, pin sclk, pin cs, pin LCD_reset, pin LCD_backlight
     */
    GLCD( PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rst, PinName bklgh);
    
    /** Configure the SPI port with default parameters.
     *
     * @param none
     */
    void lcd_init( void);
    
    /** Initialize the LCD with an optional SPI freq. Used as FastSpeed 
     *
     * @param clock speed in Hz
     */
    void lcd_init( unsigned speedf);
    
    /** Initialize the LCD with two optionals SPI freq: fast and slow speed. 
     *
     * @param speedf The fast freq. used to drive the GLCD
     * @param speeds The freq. at wich the SPI is configured after the use.
     *
     */
    void lcd_init( unsigned int speedf, unsigned int speeds);
    
    /** Clear the entire screen with color 
     *
     * #param color The color we want to use to fill the screen
     */
    void lcd_clear(unsigned int color);
    
    /** Clear a specified screen region. Like as a rectangle fill
     * @param x0 start x position
     * @param y0 start y position
     * @param w  rectangle width
     * @param h  rectangle hight
     * @param color The color we use to fill the area.
     *
     */
    void lcd_clear(unsigned int x0, unsigned int y0, unsigned int w, unsigned int h, unsigned int color);
    
#ifdef _USE_FILE    
    /** Draw an RGB image file (320x240) starting at x=0, y=0
     *  The image is read byte per byte. So use this function if you don't care about speed
     *
     * @param fname The file name we want to display
     */
    unsigned int lcd_drawimage(char *fname);
    
    /** Draw an RGB image file (320x240) starting at x=0, y=0
     *  The image is read using a buffer. You can set the buffer using the define: BUFFER_LINE
     *  This procedure speedup the draw of the image. At the cost of more RAM.
     *  The defaullt buffer size is: 320*BUFFER_LINE*3
     *
     * @param fname The file name we want to display
     */    
    unsigned int lcd_drawimagebuff(char *fname);
    
    /** Draw an image file (320x240) staring at x=0, y=0
     *  The image is displayed line-by-line so the main program can do other jobs between the call.
     *  The buffer size is equivalent to a LCD line, so must be: 320x3 bytes.
     * 
     * @param buffer The pointer at the data to be displayed
     * @param lnum   The number of the line to display. From 0 to 239.
     */
    unsigned int lcd_drawimagebyline(unsigned char *buffer, unsigned int lnum);
    
    /** Draw an image file (less then 320x240) specifying the x and y position and the scale value.
     *  The image is displayed line-by-line so the main program can do other job during the call.
     *  It is possible to specify a x and y starting position and a scale value.
     *  The buffer size is equivalent to a LCD line, so must be: 320x3 bytes.
     * 
     * @param buffer The pointer at the data to be displayed
     * @param lnum   The number of the line to display. From 0 to 239.
     * @param xstart x position 
     * @param ystart y position
     * @param scale  Scale factor value. There are two value defined: SCALE_320x240 and SCALE_160x120
     *               Using the second value it's possible to draw, from a 320x240 image, a 160x120 image.
     *               at the xy specified. This is the only value supported for now.
     */
    unsigned int lcd_drawimagebyline(unsigned char *buffer, unsigned int lnum, unsigned int xstart, unsigned int ystart, unsigned int scale);

    /** Play a movie file 80x60 pixel.
     *  The file is a ready-made array of pictures extracted from a movie, resized to 80x60 and converted to RGB format.
     *  I use this procedure in a Ubuntu Linux PC:
     *
     *  @code
     *  mplayer -endpos 10 -nosound -vo png:z=0 perla.avi
     *  mogrify -resize 80x60 *.png
     *  mogrify -format rgb *.png
     *  cat *.rgb > unico_80x60.bin
     *  @endcode
     *
     *  First I convert 10 sec of my avi files with mplayer, then I use mogrify to resize the 150 images to 80x60 pixel.
     *  The next step is to convert the PNG format to RGB. At this point I create a single file using a simple cat command.
     *  With this little format there is enough speed to play the movie at 15fps.
     */
    unsigned int lcd_drawmovie(char *fname, unsigned int x_start, unsigned int y_start);
    
    /** Play a 160x120 pixel ready made movie file.
     *  The file is a ready-made array of pictures extracted from a movie, resized to 160x120 pixel and converted to RGB format.
     *  I use this procedure in a Ubuntu Linux PC:
     *
     *  @code
     *  mplayer -endpos 10 -nosound -vo png:z=0 perla.avi
     *  mogrify -resize 160x120 *.png
     *  mogrify -format rgb *.png
     *  cat *.rgb > unico_160x120.bin
     *  RGBconv_tst unico_160x120.bin unico_160x120_lcd.bin
     *  @endcode
     *
     *  First I convert 10 sec of my avi files with mplayer, then I use mogrify to resize the 150 images generated to 160x120 pixel.
     *  The next step is to convert the PNG format to RGB. At this point I create a single file using a simple cat command.
     *  To speed up the playing procedure, I convert the RGB 24bit file to the LCD format RGB 565bit using a simple programm
     *  I create.
     *  The procedure use the USB and Ethernet memory area as a ping pong buffer. The buffer loaded from the SDCard is send to the LCD using the DMA.
     *  During this time another buffer is read and stored to the second buffer. The single DMA transfer is too short for me and I use 3 linked list 
     *  to chain 3 buffer of memory and send simultaneously more then 12KB if data.
     *
     *  @param fname the file name to display
     *  @param x_start and y_start, the x and y position where to display the movie.
     */
    unsigned int lcd_drawmoviebuff(char *fname, unsigned int x_start, unsigned int y_start);
    
#endif    // End of: #ifdef _USE_FILE

    /** Draw an RGB icon at x, y. The size is specified for width and hight.
     *
     * @param *icon  pointer at the array with the RGB value in integer format. That is: 0x00RRGGBB
     * @param x      x position
     * @param y      y position
     * @param size   the width and height
     */
    void lcd_drawicon( const unsigned int *icon, unsigned int x, unsigned int y, unsigned int size);

    /** Draw an RGB icon at x, y. The size is specified for width and hight.
     *
     * @param *icon  pointer at the array with the RGB value in byte format. That is: 0xRR, 0xGG, 0xBB
     * @param x      x position
     * @param y      y position
     * @param size   the width and height
     */    
    void lcd_drawicon( const unsigned char *icon, unsigned int x, unsigned int y, unsigned int size);
    
    /** Draw an RGB icon at x, y. The size is specified for width and hight.
     *
     * @param *icon  pointer at the array with the RGB value in byte format. That is: 0xRR, 0xGG, 0xBB
     * @param x      x position
     * @param y      y position
     * @param xsize   the width
     * @param ysize   the height
     */     
    void lcd_drawicon( const unsigned char *icon, unsigned int x, unsigned int y, unsigned int xsize, unsigned int ysize);
    
    /** Draw a color line from xy to xy
     * 
     * @param x0  x position
     * @param y0  y position
     * @param x1  x position
     * @param y1  y position     
     * @param color  color to use
     */
    void lcd_drawline(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1, unsigned int color);
    
    /** Draw a single pixel
     *
     * @param x0  x position
     * @param y0  y position
     * @param color  color to use
     */
    void lcd_drawpixel(unsigned int x0, unsigned int y0, unsigned int color);
    
    /** Draw a character at xy position.
     * 
     * @param ch    the character to draw
     * @param xpos  x position
     * @param ypos  y position 
     * @param color the color to use
     */
    void lcd_drawch( unsigned ch, unsigned int xpos, unsigned int ypos, unsigned int color);
    
    /** Draw a string at xy position 
     *
     * @param *putstr the string to draw
     * @param xpos    x position
     * @param ypos    y position
     * @param color   the color to use
     */
    void lcd_drawstr(char *__putstr, unsigned int xpos, unsigned int ypos, unsigned int color);
    
    /** Set the font type to small: 12x6
     */
    void lcd_setfontsmall( void);
    
    /** Set the font type to big: 24x12
     */    
    void lcd_setfontbig( void);
        
    /** Return the color value use by the LCD. There are two form: 666 and 565
     *
     * @param color   the color in the format: 0x00RRGGBB
     * @return        the color in the format: 666 or 565
     */
    unsigned int lcd_RGB( unsigned int color);
    
    /** Return the color value use by the LCD. There are two form: 666 and 565
     *
     * @param r   the Red color value
     * @param g   the Green color value
     * @param b   the Blue color value
     * @return        the color in the format: 666 or 565
     */    
    unsigned int lcd_RGB( unsigned int r, unsigned int g, unsigned int b);
    
    /** Set the background color
     * 
     * @param color  the color of the background
     */
    void lcd_setbackgroundcolor( unsigned int color);

    /** Function to start the automated scroll of the screen
     */
    void lcd_setverticalarea( unsigned int topf, unsigned int height, unsigned int butf);
    void lcd_scrollstartadd( unsigned int ssa);
    void lcd_scrollstart( void);
    void lcd_scrollstop( void);    
    
    /** Set the backlight ON
     *
     */
    void backlighton( void);
    
    /** Set the backlight OFF
     *
     */    
    void backlightoff( void);

    /** Set the backlight at a particular value.
     *
     * @param val  backlight value to set.
     */    
    void backlightset( double val);
        
protected:
    void lcd_draw(unsigned int color);
    void lcd_drawstop(void);
    void lcd_drawstart(void);
    void lcd_cmd(unsigned int reg, unsigned int param);
    void lcd_data(unsigned int c);
    void lcd_area(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1);
    void lcd_reset(void);
    
    // default value for SPIClock speed, SPI mode and backlight freq.
    unsigned int SPIClkSpeed;
    unsigned int SPISlowClkSpeed;
    unsigned int SPIMode;
    double       LCDBkLight;
    // default value for background color
    unsigned int BackGroundColor;
    //
    unsigned char FontIdx;
    _FONTINFO FontInfo[2];
    
    SPI         _spi;
    DigitalOut  _cs;
    DigitalOut  _rst;
    PwmOut      _bklgh;      
    
};

#endif