/*---------------------------------------------------------------------------

    QRSS Receiver Application
        
    by Clayton ZL3TKA/VK1TKA
    clayton@isnotcrazy.com

    main function

---------------------------------------------------------------------------*/
// include files

#include "mbed.h"
#include "global.h"
#include "EthernetNetIf.h"
#include "UDPSocket.h"
#include "dnsresolve.h"
#include "gps.h"
#include "BufferSys.h"
#include "I2S_Rx.h"
#include "DSP.h"
#include "comms.h"

// Definitions

// Macros

// Local Data

// Global Data

// Serial port interface to the PC
Serial pc( USBTX, USBRX );

// LEDs
DigitalOut OnLED( LED1 );

// IO
DigitalOut TestOscillator( p21 );
#define TESTOSC_ON      0
#define TESTOSC_OFF     1

// The Buffer pool for audio data
TBufferPool     BufferPool;

// The array of buffers for the pool
TBufferData BufferDataArray[NUM_OF_BUFFERS];

// I2S Object
TI2SReceiver    AudioInput( BufferPool );

// Communications Object
TCommunications Comms;


// Function Prototypes

//---------------------------------------------------------------------------
//  MAIN
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
//  Main start routine
//      Just starts everything running
//
int main()
{
    TDSPProcessor   ADCBuffer;
    uint32_t        uiCurrentTestmode = 0x1234;

    // Set up IOs
    OnLED = 1;
    TestOscillator = TESTOSC_OFF;

    // Set up PC port
    pc.baud( 38400 );

    // Report
    printf( "\r\n\r\nQRSS_Rx Application running\r\n" );
    printf( "Built " __DATE__ "   " __TIME__ "\r\n" );
    printf( "System Core Clock = %ld\r\n", SystemCoreClock );

    // Add array of buffers to the pool
    pc.printf ("Setting up buffer pool with %d buffer of %d samples ...", NUM_OF_BUFFERS, BUFFERSYS_SIZE );
    BufferPool.AddNewMsgsToPool( BufferDataArray, NUM_OF_BUFFERS );
    printf (" OK\r\n");

    // Set up audio receiver
    pc.printf ("Initialise the I2S Receiver ...");
    AudioInput.Init();
    printf (" OK\r\n");

    // Set up GPS
    pc.printf ("Initialise GPS ...");
    GPSModule.Init();
    printf (" OK\r\n");

    // Set up communications
    pc.printf ("Initialise Communications ...");
    Comms.Init();
    Comms.SetServer( "192.168.0.15" );      // default QRSS Server Address (could be a DNS name)
    printf (" OK\r\n");

    ADCBuffer.NCOFrequency( TEST_NCO_FREQ );

    // echo GPS data out the PC port
    while(1) 
    {
        // Poll the sub-systems
        Net::poll();
        GPSModule.Poll();
        Comms.Poll();

        // Handle Comms Mode
        if ( Comms.Running() )
        {   // online comms
            if ( !AudioInput.Running() )
            {
                printf( "Starting Processing\r\n" );
                ADCBuffer.NCOPhaseInc( Comms.NCOPhaseInc() );
                AudioInput.Start();
            }
            // process and output any received samples
            if ( !AudioInput.Empty() )
            {
                // ADC buffer will be automatically released
                AudioInput.Read( ADCBuffer );
                // Mix the LO
                ADCBuffer.MixLO();
                // Filter the result
                ADCBuffer.LPF();
                // Output the data to the comms module
                for ( int ii=0; ii<8; ii++ )
                    Comms.SendSamples( ADCBuffer.Timestamp(), ADCBuffer.FilteredOutput(ii) );
            }
        }
        else
        {   // offline comms
            if ( AudioInput.Running() )
            {
                AudioInput.Stop();
                printf( "Stopping Processing\r\n" );
            }
            if ( !AudioInput.Empty() )
            {   // discard messages
                AudioInput.Read( ADCBuffer );
                ADCBuffer.Release();
            }
        }

        // Handle Test Modes
        if ( Comms.TestMode()!=uiCurrentTestmode )
        {
            uiCurrentTestmode = Comms.TestMode();
            if ( uiCurrentTestmode!=0 )
            {   // enable test Osc
                printf( "Turn ON Test Osc\r\n" );
                TestOscillator = TESTOSC_ON;
            }
            else
            {   // disable test Osc
                printf( "Turn OFF Test Osc\r\n" );
                TestOscillator = TESTOSC_OFF;
            }
        }

        // Process Keystrokes - debug only
        if( pc.readable() )
        {
            int iCharIn = pc.getc();
            switch ( iCharIn )
            {
              case'1':
                printf( "Turn on Test Osc\r\n" );
                TestOscillator = TESTOSC_ON;
                break;
              case '2':
                printf( "Turn off Test Osc\r\n" );
                TestOscillator = TESTOSC_OFF;
                break;
              case '7':
                printf( "AudioInput Start\r\n" );
                AudioInput.Start();
                break;
              case '8':
                printf( "AudioInput Stop\r\n" );
                AudioInput.Stop();
                break;
              case '9':
                AudioInput.Report();
                break;
              case 'x':
                ADCBuffer.NCOFrequency( 15000 );
                break;
              case 'y':
                ADCBuffer.NCOFrequency( -15000 );
                break;
              case 'z':
                ADCBuffer.NCOFrequency( 15600 );
                break;
            }
        }
    }
        
}

//---------------------------------------------------------------------------
//  END
//---------------------------------------------------------------------------


