/*---------------------------------------------------------------------------

    QRSS Receiver Application
        
    by Clayton ZL3TKA/VK1TKA
    clayton@isnotcrazy.com

    Header File for I2S Receiver
        
    I2S Receiver operates in slave mode

---------------------------------------------------------------------------*/
#ifndef _I2S_RX_H
#define _I2S_RX_H

#include "mbed.h"
#include "BufferSys.h"

// Definitions

#define I2S_DUMMY_BUFFER_SIZE       16

// Macros

//
// Classes
//

//---------------------------------------------------------------------------
//
//  I2S Receiver Class
//
class TI2SReceiver
{
    // create/destroy
    public:
        TI2SReceiver( TBufferPool &BuffPol );
        ~TI2SReceiver() 
            {
                Stop();
            }

    // API
    public:
        // Initialise the hardware
        void Init();

        // Start receiving
        void Start();
        
        // Stop receiving
        void Stop();
        
        // Receiver status
        bool Running()
            { return bRunning; }
        
        // Test for a buffer available
        //  Return number of queued buffers
        int Count() const
            { return ReceivedBuffers.Count(); }
        //  Test if empty
        bool Empty() const
            { return ReceivedBuffers.Empty(); }

        // Read out a buffer
        bool Read( TBufferHandle &Handle )
            {
                // queue is filled from interrupts, so we need to ensure we always access it with interrupts off
                __disable_irq();
                bool bRet = ReceivedBuffers.Read(Handle);
                __enable_irq();
                return bRet;
            }

        // report the number of buffer failures
        uint32_t BufferFailures() const
            { return ulBufferFailures; }

        // DMA IRQ Routine
        void DMA_Interrupt(void);

        // Debug routine - 
        // Read samples from the I2S by polling it
        //  Returns number of samples read
        int PolledRead( int32_t *piBuffer, int iLen );

        //  Debug routine - 
        //      Read status register
        uint32_t Status();

        //  Debug routine - 
        //      Report Status
        void Report();

    // Private methods
    private:

        //  Start a new DMA transfer
        void StartDMATransfer();

    // data
    private:
        // pool to get buffers from
        TBufferPool     &EmptyBuffersPool;

        // queue of incoming sample buffers
        TBufferQueue    ReceivedBuffers;

        // the current buffer being used
        TBufferHandle   CurrentRxBuffer;

        // flags indicating run status
        bool            bRunning;

        // a buffer to use when there are no empty buffers available
        int32_t        aulDummyBuffer[I2S_DUMMY_BUFFER_SIZE];
        
        // frame counters etc
        uint32_t        ulBufferCount;
        uint32_t        ulBufferAllocations;
        uint32_t        ulBufferFailures;
        
};

#endif

//---------------------------------------------------------------------------
//  END
//---------------------------------------------------------------------------

