/*---------------------------------------------------------------------------

    QRSS Receiver Application
        
    by Clayton ZL3TKA/VK1TKA
    clayton@isnotcrazy.com

    Header File for Communications Module

---------------------------------------------------------------------------*/
#ifndef _COMMS_H
#define _COMMS_H

#include "mbed.h"
#include "global.h"
#include "EthernetNetIf.h"
#include "UDPSocket.h"
#include "dnsresolve.h"
#include "BufferSys.h"
#include "gps.h"

// Definitions

// UDP output data size - includes header
#define COMMS_DATABUFF_HEADER   (4+4)
#define COMMS_DATABUFF_SIZE     ((LPF_OUTPUTS_SIZE*SAMPLE_SET_SIZE*8)+COMMS_DATABUFF_HEADER)

// UDP Control Buffer Size
#define COMMS_CNTRLBUFF_SIZE    (60)

// Number of seconds between each poll message
#define POLL_MSG_PERIOD         (6)

// default poll period in absence of 1PPS (mSec)
#define POLL_TIMEOUT            (10000)

// time to stop sending data if no commands are received
#define RUNNING_TIMEOUT         (30000)

//
//  Message Encoding
//
#define COMM_POLL_MSG               10
#define COMM_CMND_MSG               11
#define COMM_DATA_MSG               12
#define COMM_MSG_VERSION            1
#define COMM_MSG_SW_VERSION         1
#define COMM_MSG_HW_ID              1
#define COMM_MSG_REF_CLOCK          96000000
#define COMM_MSG_RF_MIXER           10125000
#define COMM_MSG_SAMPLE_CLOCK       13500000
#define COMM_MSG_SAMPLE_DIVIDER     (384*64)
#define COMM_MSG_NAME               "QRSS_proto"
#define COMM_MSGNAME_LEN            10

#define COMM_CMND_MSG_LENGTH        (5*4)

// Macros

//
// Classes
//

//---------------------------------------------------------------------------
//
//  Communications Class
//
class TCommunications
{
    // create/destroy
    public:
        TCommunications() :
            EthernetUpLED( LED2 ),
            RunningLED( LED4 )
            {}
        ~TCommunications() 
            {}

    // Local Types - state machine state
    typedef enum {
        ST_START_ETH,
        ST_POLL_SERVER,
        ST_RUNNING
        } TCommsStates;

    // API
    public:
        // Initialisation
        void Init();

        // Set up Server DNS name
        void SetServer( char *szNm )
            { 
                strncpy( szServerName, szNm, sizeof(szServerName) ); 
                szServerName[sizeof(szServerName)-1] = 0;   // place terminator at end incase name was too long
            }

        // Processing routine
        void Poll();

        //  RecordPPSTimestamps Method
        //      Records timestamps of events (1PPS or capture of LO divider)
        void RecordPPSTimestamps( uint32_t uiPPSCapture, uint32_t uiLOscCapture, 
                                  TGPSController::TGPSData &GPSInfo );

        // Test if comms is ready for sample data (ie in RUNNING state)
        bool Running()
            { return (eState==ST_RUNNING); }

        // Return NCO Inc setting received
        uint32_t NCOPhaseInc()
            { return uiNCOPhaseInc; }

        // Return Test Mode setting received
        uint32_t TestMode()
            { return uiTestModeInc; }

        //  Pass sample data for sending. Send when buffer is full
        void SendSamples( uint32_t uiTimestamp, TDataSample sSample );


    // Private methods
    private:
        // Callback for incoming UDP data
        void onUDPSocketEvent( UDPSocketEvent evnt );

        // Send a UDP Poll Message
        bool SendPollMessage();

        // Place a word into a buffer - network byte order
       void WriteWord( uint8_t * &pucBuff, int32_t iWord, int &iLn )
            {
                *pucBuff = (uint8_t)(iWord>>24);
                pucBuff++;
                *pucBuff = (uint8_t)(iWord>>16);
                pucBuff++;
                *pucBuff = (uint8_t)(iWord>>8);
                pucBuff++;
                *pucBuff = (uint8_t)(iWord>>0);
                pucBuff++;
                iLn += 4;
            }

        // Read a word from a buffer - network byte order
       uint32_t ReadWord( uint8_t * pucBuff )
            { return (0x1000000*pucBuff[0]) | (0x10000*pucBuff[1]) | (0x100*pucBuff[2]) | (pucBuff[3]); }

    // data
    private:
        // LEDs
        DigitalOut EthernetUpLED;
        DigitalOut RunningLED;

        // State machine state
        TCommsStates    eState;

        // Ethernet Interface
        EthernetNetIf   eth;
        
        // UDP Socket for comms
        char            szServerName[50];
        UDPSocket       ServerSocket;
        Host            MyPort;
        Host            ServerPort;
        IpAddr          ServerAddr;
        
        // DNS lookup
        DNSResolver     dnsService;
        
        // Host Details
        
        // Output buffers - word align this buffer
        uint32_t        auiDataBuffer[(COMMS_DATABUFF_SIZE/4)+8];   // the buffer - defined as words to ensure alignment
        uint8_t        *pucBuffDataPtr;     // buffer pointer - as a byte pointer
        int             iBufferCount;       // byte count

        // Incoming UDP Data
        uint32_t        uiCommand;
        uint32_t        uiNCO;
        uint32_t        uiTestOsc;
        bool            bGotCmnd;

        // PPS Timer Event Counter
        int             iPPSCount;
    
        // Send Poll Message timer
        Timer           PollTimer;
        Timer           CommandTimer;
        
        // Timer Capture Data - queue of 8 captures. [0] is the most recent
        uint32_t        auiPPSCaptures[8];
        uint32_t        auiLOscCaptures[8];

        // Last GPS info        
        TGPSController::TGPSData    LastGPSInfo;
        
        // data from the command packet
        uint32_t        uiNCOPhaseInc;
        uint32_t        uiTestModeInc;

};

// declare the Communications module
extern TCommunications Comms;

#endif

//---------------------------------------------------------------------------
//  END
//---------------------------------------------------------------------------

