// ----------------------------------------------------------------------------
// Copyright 2016-2017 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------

#include "mbed.h"
#include "mbed-trace/mbed_trace.h"
#include "mbed-trace-helper.h"
#include "simple-mbed-cloud-client.h"
#include "key-config-manager/kcm_status.h"
#include "key-config-manager/key_config_manager.h"
#include "SDBlockDevice.h"
#include "FATFileSystem.h"
#include "EthernetInterface.h"

// Placeholder to hardware that trigger events (timer, button, etc)
Ticker timer;
InterruptIn sw2(SW2);
DigitalOut led(LED1,1);

// Placeholder for storage
SDBlockDevice sd(D11, D12, D13, D10);   //MOSI, MISO, SCLK, CS
FATFileSystem fs("sd");

// To-Do #2: Add global pointer declaration for Rate Resource
// Pointer declaration for Rate Resource
static MbedCloudClientResource* rate_ptr;

static bool button_pressed = false;
void button_press() {
    button_pressed = true;
}

void led_toggle() {
    led = !led;
}

// To-Do #3: Add Cloud Client Resource Callback Functions
// Resource Callback Functions
void blink_rate_updated(const char *) {
    printf("PUT received, Storing LED Blink Rate: %s (ms)\r\n", rate_ptr->get_value().c_str());
}

void blink_enable_callback(void *) {
    String pattern_str = rate_ptr->get_value();
    const char *rate = pattern_str.c_str();
    printf("POST received. Enabling LED Blink Rate = %s (ms)\n", rate);

    float value = atoi(rate_ptr->get_value().c_str())/1000.0;
    timer.detach();
    timer.attach(&led_toggle,value);
}

void button_callback(const M2MBase& object, const NoticationDeliveryStatus status)
{
    printf("Button notification. Callback: (%s)\n", object.uri_path());
}


int main(void)
{
    // Requires DAPLink 245+ (https://github.com/ARMmbed/DAPLink/pull/364)
    // Older versions: workaround to prevent possible deletion of credentials:
    wait(2);

    // Misc OS setup
    srand(time(NULL));

    // Placeholder for network
    EthernetInterface net;

    printf("Start Simple Mbed Cloud Client\n");

    // Initialize SD card
    int status = sd.init();
    if (status != BD_ERROR_OK) {
        printf("Failed to init SD card\r\n");
        return -1;
    }

    // Mount the file system (reformatting on failure)
    status = fs.mount(&sd);
    if (status) {
        printf("Failed to mount FAT file system, reformatting...\r\n");
        status = fs.reformat(&sd);
        if (status) {
            printf("Failed to reformat FAT file system\r\n");
            return -1;
        } else {
            printf("Reformat and mount complete\r\n");
        }
    }

    // Connect to Network (obtain IP address)
    printf("Connecting to the network using Ethernet...\n");
    status = net.connect();
    if (status) {
        printf("Connection to Network Failed %d!\n", status);
        return -1;
    } else {
        const char *ip_addr  = net.get_ip_address();
        printf("Connected successfully\n");
        printf("IP address %s\n", ip_addr);
    }

    // Mbed Cloud Client Initialization
    SimpleMbedCloudClient mbedClient(&net);
    status = mbedClient.init();
    if (status) {
        return -1;
    }
    printf("Client initialized\r\n");

    // To-Do #1: Add Mbed Cloud Client resources
    // Setup Mbed Cloud Client Resources
    MbedCloudClientResource *button = mbedClient.create_resource("3200/0/5501", "button_resource"); // Digital Input / Instance / Counter
    button->set_value("0");
    button->methods(M2MMethod::GET);
    button->observable(true);
    button->attach_notification_callback(button_callback);
    
    MbedCloudClientResource *rate = mbedClient.create_resource("3201/0/5521", "blink_rate_resource"); // Digital Output / Instance / Delay Duration
    rate->set_value("500");
    rate->methods(M2MMethod::GET | M2MMethod::PUT);
    rate->observable(false);
    rate->attach_put_callback(blink_rate_updated);
    rate_ptr = rate;
    
    MbedCloudClientResource *blink = mbedClient.create_resource("3201/0/5823", "blink_enable_resource"); // Digital Output / Instance / Event ID
    blink->methods(M2MMethod::POST);
    blink->attach_post_callback(blink_enable_callback);

    // Mbed Cloud Register Client and Connect
    mbedClient.register_and_connect();

    // Wait for client to finish registering
    while (!mbedClient.is_client_registered()) {
        wait_ms(100);
    }

    // Setup LED and Push BUtton
    timer.attach(&led_toggle, 0.5);
    sw2.fall(&button_press);

    // Check if client is registering or registered, if true sleep and repeat.
    while (mbedClient.is_register_called()) {
        static int button_count = 0;
        wait_ms(100);
        
        if (button_pressed) {
            button_pressed = false;
            printf("Button Pressed %d time(s).\r\n",(++button_count));
            // To-Do #4: Add function to set button resource value
            // Call to update button resource count
            button->set_value(button_count);
        }
    }

    // Client unregistered, exit program.
    return 0;
}
