
#include "RGBA_LED.h"

RGBA_LED::RGBA_LED(I2C *i2c, DigitalOut *inv_out_en, int addr, bool amber_enabled)
: step_time_(RGB_LED_PERIOD), def_color_fade_time_(DEF_CLR_FADE_T), def_alpha_fade_time_(DEF_ALP_FADE_T)
{
    
    led_driver_ = new PCA9634(i2c, inv_out_en, addr);
    
    strip_leds_[0] = RGBA<ledID>(LED_STR0_R, LED_STR0_G, LED_STR0_B, LED_STR0_A);
    strip_leds_[1] = RGBA<ledID>(LED_STR1_R, LED_STR1_G, LED_STR1_B, LED_STR1_A);
    
    setColor(RGBA<float>(0,0,0,0));
    setBrightness(1.0);
    
    // TODO: Replace with setColorProgram, setBrightnessProgram
    for (int i = 0; i < 2; i++) {
        count_color_[i] = 0;
        count_alpha_[i] = 0;
    }
    
    if (amber_enabled)
        enableAmber();
    else
        disableAmber();
}

void RGBA_LED::disableAmber() {
    disableAmber(0);
    disableAmber(1);
}
void RGBA_LED::disableAmber(int stripID) {
    led_driver_->disableLED(strip_leds_[stripID].a);
}

void RGBA_LED::enableAmber() {
    enableAmber(0);
    enableAmber(1);
}
void RGBA_LED::enableAmber(int stripID) {
    led_driver_->enableLED(strip_leds_[stripID].a);
}


void RGBA_LED::setColor(RGBA<float> color) {
    setColor(color, 0);
    setColor(color, 1);
}

void RGBA_LED::setColor(RGBA<float> color, int stripID) {
    cur_color_[stripID] = color;
    updatePWMs(stripID);
}

void RGBA_LED::setBrightness(float alpha) {
    setBrightness(alpha, 0);
    setBrightness(alpha, 1);
}

void RGBA_LED::setBrightness(float alpha, int stripID) {
    cur_alpha_[stripID] = alpha;
    updatePWMs(stripID);
}


void RGBA_LED::setDefaultColorFadeTime(int def_color_fade_time) {
    def_color_fade_time_ = def_color_fade_time;
}
    
void RGBA_LED::setDefaultBrightnessFadeTime(int def_alpha_fade_time) {
    def_alpha_fade_time_ = def_alpha_fade_time;
}

int RGBA_LED::getStepTime() {
    return step_time_;
}

void RGBA_LED::step() {
    bool update = false;
    
    for (int i = 0; i < 2; i++) {
        if (count_color_[i] > 0) {
            count_color_[i]--;
            // Transition to next color in program
            if (count_color_[i] == 0) {                
                int i_curr = (ind_color_program_[i] + 1) % color_program_[i].size();
                int i_next = (ind_color_program_[i] + 2) % color_program_[i].size();
                ind_color_program_[i] = i_curr;
                cur_color_[i] = color_program_[i][i_curr];
                count_color_[i] = ceil(color_time_program_[i][i_curr]/step_time_);
                dcolor_[i] = (color_program_[i][i_next] - color_program_[i][i_curr]) / count_color_[i];
            } else
                cur_color_[i] += dcolor_[i];
            update = true;
        }
        if (count_alpha_[i] > 0) {
            count_alpha_[i]--;
            // Transition to next alpha in program
            if (count_alpha_[i] == 0) {                
                int i_curr = (ind_alpha_program_[i] + 1) % alpha_program_[i].size();
                int i_next = (ind_alpha_program_[i] + 2) % alpha_program_[i].size();
                ind_alpha_program_[i] = i_curr;
                cur_alpha_[i] = alpha_program_[i][i_curr];
                count_alpha_[i] = ceil(alpha_time_program_[i][i_curr]/step_time_);
                dalpha_[i] = (alpha_program_[i][i_next] - alpha_program_[i][i_curr]) / count_alpha_[i];
            } else
                cur_alpha_[i] += dalpha_[i];
            update = true;
        }
        if (update) {
            updatePWMs(i);
            update = false;
        }
    }        
        
}

void RGBA_LED::setColorProgram(vector<RGBA<float> > color_program,
    vector<float> time_program, bool fadeInto = false) {
    setColorProgram(color_program, time_program, 0, fadeInto);
    setColorProgram(color_program, time_program, 1, fadeInto);
}

void RGBA_LED::setColorProgram(vector<RGBA<float> > color_program,
    vector<float> time_program, int stripID, bool fadeInto = false) {
    color_program_[stripID] = color_program;
    color_time_program_[stripID] = time_program;
    
    // Begin transition to the first color, using default color fade time
    if (fadeInto) {     
        ind_color_program_[stripID] = color_program.size()-1;
        count_color_[stripID] = ceil(def_color_fade_time_/step_time_);
        dcolor_[stripID] = (color_program[0] - cur_color_[stripID]) / count_color_[stripID];
        
    // Switch to first color, and let step() begin transition to next color
    } else {    
        ind_color_program_[stripID] = 0;
        count_color_[stripID] = 1;
        cur_color_[stripID] = color_program[0];
        updatePWMs();
    }
}

RGBA<float> RGBA_LED::getPresetColor(char colorID) {
    switch (colorID) {
        case 'w': return RGBA<float>(255, 255, 255,   0);   // White
        case 'r': return RGBA<float>(255,   0,   0,   0);   // Red
        case 'g': return RGBA<float>(  0, 255,   0,   0);   // Green
        case 'b': return RGBA<float>(  0,   0, 255,   0);   // Blue
        case 'y': return RGBA<float>(255, 255,   0,   0);   // Yellow
        case 'c': return RGBA<float>(  0, 255, 255,   0);   // Cyan
        case 'm': return RGBA<float>(255,   0, 255,   0);   // Magenta
        default: return RGBA<float>(0, 0, 0, 0);
        // TODO: Add orange, purple
    }
}

vector<RGBA<float> > RGBA_LED::getPresetColorProgram(int programID) {
    /*switch (colorID) {
        case 0: return RGBA<vector<float> >(getPresetColor('r'), getPresetColor('g'), getPresetColor('b')); 
        default: return RGBA<vector<float> >(getPresetColor('w'));
    }*/
}


void RGBA_LED::updatePWMs() {
    updatePWMs(0);
    updatePWMs(1);
}

void RGBA_LED::updatePWMs(int stripID) {
    for (int i = 0; i < 4; i++)
        led_driver_->commandLEDBrightness(strip_leds_[stripID][i], (int) (cur_color_[stripID][i]*cur_alpha_[stripID]));
}