#ifndef RGBA_H
#define RGBA_H

#include <iostream>
#include <cmath>

#define for_i for(std::size_t i = 0; i != 4; ++i)

template <typename T>
struct RGBA
{
  union {
    struct {
      T r;
      T g;
      T b;
      T a;
    };
    T elem[4];
  };

  // CONSTRUCTORS

  RGBA() {
    for_i elem[i] = T();
  }
  explicit RGBA(T b) {
    for_i elem[i] = b;
  }
  RGBA(T b0, T b1, T b2, T b3) {
    elem[0] = b0; elem[1] = b1; elem[2] = b2; elem[3] = b3;
  }

  // MODIFIERS

  /** Add scalar @a b to this RGBA */
  RGBA& operator+=(T b) {
    for_i elem[i] += b;
    return *this;
  }
  /** Subtract scalar @a b from this RGBA */
  RGBA& operator-=(T b) {
    for_i elem[i] -= b;
    return *this;
  }
  /** Scale this RGBA up by scalar @a b */
  RGBA& operator*=(T b) {
    for_i elem[i] *= b;
    return *this;
  }
  /** Scale this RGBA down by scalar @a b */
  RGBA& operator/=(T b) {
    for_i elem[i] /= b;
    return *this;
  }
  /** Add RGBA @a b to this RGBA */
  RGBA& operator+=(const RGBA& b) {
    for_i elem[i] += b[i];
    return *this;
  }
  /** Subtract RGBA @a b from this RGBA */
  RGBA& operator-=(const RGBA& b) {
    for_i elem[i] -= b[i];
    return *this;
  }
  /** Scale this RGBA up by factors in @a b */
  RGBA& operator*=(const RGBA& b) {
    for_i elem[i] *= b[i];
    return *this;
  }
  /** Scale this RGBA down by factors in @a b */
  RGBA& operator/=(const RGBA& b) {
    for_i elem[i] /= b[i];
    return *this;
  }

  // ACCESSORS

  T&       operator[](std::size_t i)       { return elem[i]; }
  const T& operator[](std::size_t i) const { return elem[i]; }

  T*        data()       { return elem; }
  const T*  data() const { return elem; }

  T&       front()       { return elem[0]; }
  const T& front() const { return elem[0]; }
  T&       back()        { return elem[3]; }
  const T& back()  const { return elem[3]; }

  std::size_t     size() { return 4; }
  std::size_t max_size() { return 4; }
  bool         empty() { return false; }

  // ITERATORS

  T*        begin()       { return elem; }
  const T*  begin() const { return elem; }
  const T* cbegin() const { return elem; }

  T*          end()       { return elem+4; }
  const T*    end() const { return elem+4; }
  const T*   cend() const { return elem+4; }
};

// STREAM OPERATORS

/** Write a RGBA to an output stream */
template <typename T>
std::ostream& operator<<(std::ostream& s, const RGBA<T>& a) {
  return (s << a.r << ' ' << a.g << ' ' << a.b ' ' << a.a);
}
/** Read a Vec from an input stream */
template <typename T>
std::istream& operator>>(std::istream& s, RGBA<T>& a) {
  return (s >> a.r >> a.g >> a.b >> a.a);
}

// COMPARATORS
template <typename T>
bool operator==(const RGBA<T>& a, const RGBA<T>& b) {
  return std::equal(a.begin(), a.end(), b.begin());
}
template <typename T>
bool operator!=(const RGBA<T>& a, const RGBA<T>& b) {
  return !(a == b);
}

// ARITHMETIC OPERATORS

/** Unary negation: Return -@a a */
template <typename T>
RGBA<T> operator-(const RGBA<T>& a) {
  return RGBA<T>(-a.r, -a.g, -a.b, -a.a);
}
/** Unary plus: Return @a a. ("+a" should work if "-a" works.) */
template <typename T>
RGBA<T> operator+(const RGBA<T>& a) {
  return a;
}
template <typename T>
RGBA<T> operator+(RGBA<T> a, const RGBA<T>& b) {
  return a += b;
}
template <typename T>
RGBA<T> operator+(RGBA<T> a, double b) {
  return a += b;
}
template <typename T>
RGBA<T> operator+(double b, RGBA<T> a) {
  return a += b;
}
template <typename T>
RGBA<T> operator-(RGBA<T> a, const RGBA<T>& b) {
  return a -= b;
}
template <typename T>
RGBA<T> operator-(RGBA<T> a, double b) {
  return a -= b;
}
template <typename T>
RGBA<T> operator-(double b, const RGBA<T>& a) {
  return (-a) += b;
}
template <typename T>
RGBA<T> operator*(RGBA<T> a, const RGBA<T>& b) {
  return a *= b;
}
template <typename T>
RGBA<T> operator*(RGBA<T> a, double b) {
  return a *= b;
}
template <typename T>
RGBA<T> operator*(double b, RGBA<T> a) {
  return a *= b;
}
template <typename T>
RGBA<T> operator/(RGBA<T> a, const RGBA<T>& b) {
  return a /= b;
}
template <typename T>
RGBA<T> operator/(RGBA<T> a, double b) {
  return a /= b;
}

#undef for_i

#endif
