/*

This is a library for the BH1750FVI (GY-30) Digital Light Sensor
breakout board.

The board uses I2C for communication.

Datasheet:
http://rohmfs.rohm.com/en/products/databook/datasheet/ic/sensor/light/bh1750fvi-e.pdf

Inspired by Michal Stehlik, August, 2014.
Written by Josef Křivský, January, 2018.

*/

#ifndef BH1750_h
#define BH1750_h

#include "mbed.h"

#define BH1750_I2CADDR 0x46

// Neaktivní stav
#define BH1750_POWER_DOWN 0x00

// Čekání na příkaz
#define BH1750_POWER_ON 0x01

// Reset datového registru senzoru - není akceptováno v neaktivním stavu
#define BH1750_RESET 0x07

// Start kontinuálního měření s rozlišením 1lx (trvání přibližně 120ms).
#define BH1750_CONTINUOUS_HIGH_RES_MODE  0x10

// Start kontinuálního měření s rozlišením 0.5lx (trvání přibližně 120ms).
#define BH1750_CONTINUOUS_HIGH_RES_MODE_2  0x11

// Start kontinuálního měření s rozlišením 4lx (trvání přibližně 16ms).
#define BH1750_CONTINUOUS_LOW_RES_MODE  0x13

// Start jednoho měření s rozlišením 1lx (trvání přibližně 120ms).
// Zařízení je po přečtení naměřených dat automaticky vypnuto
#define BH1750_ONE_TIME_HIGH_RES_MODE  0x20

// Start jednoho měření s rozlišením 0.5lx (trvání přibližně 120ms).
// Zařízení je po přečtení naměřených dat automaticky vypnuto
#define BH1750_ONE_TIME_HIGH_RES_MODE_2  0x21

// Start jednoho měření s rozlišením 4lx (trvání přibližně 16ms).
// Zařízení je po přečtení naměřených dat automaticky vypnuto
#define BH1750_ONE_TIME_LOW_RES_MODE  0x23

/**
* @code
* #include "mbed.h"
* #include "BH1750.h"
*
* BH1750 lightSensor(I2C_SDA, I2C_SCL);
* Serial pc(SERIAL_TX, SERIAL_RX);
*
* int main() 
* {
*    lightSensor.init();
*    lightsensor.contMeas();
*
*    while(1) {
*        pc.printf("Light intensity: %.4f Lux\r\n", lightSensor.readIntesity());
*        wait(5);
*    }
* }
* @endcode
*/

class BH1750
{
public:
    BH1750(PinName sda = I2C_SDA, PinName scl = I2C_SCL);
    void init(void); // Inicializace senzoru
    uint16_t singleMeas(char mode = BH1750_ONE_TIME_HIGH_RES_MODE); // Provedení jediného měření a vyčtení výsledku. Akceptované hodnoty jsou typu BH1750_ONE_TIME_...
    void contMeas(char mode = BH1750_CONTINUOUS_HIGH_RES_MODE); // Začátek kontinuálního měření bez vyčtení výsledků.
    void configure(char mode); // Konfigurace senzoru na specifikovaný mód
    uint16_t readIntesity(void); // Vyčtení výsledku posledního měření
    char* getRawData(void); // Vyčtení dat ze senzoru bez jejich úpravy
    int getStatus(void); // Zjištění statusu senzoru a I2C komunikace
    uint16_t getLimit(void); // Zjištění aktuálního limitu osvětlení
    void setLimit(uint16_t lim); // Nastavení limitu osvětlení
    void setCurrentLimit(void); // Nastavení limitu osvětlení na jeho aktuální hodnotu

private:
    I2C i2c;
    char rawDataBuffer[2];
    int status;
    uint16_t limit;
};

#endif
