/*

This is a library for the BH1750FVI (GY-30) Digital Light Sensor
breakout board.

The board uses I2C for communication.

Datasheet:
http://rohmfs.rohm.com/en/products/databook/datasheet/ic/sensor/light/bh1750fvi-e.pdf

Inspired by Michal Stehlik, August, 2014.
Written by Josef Křivský, January, 2018.

*/

#include "BH1750.h"

BH1750::BH1750(PinName sda, PinName scl) : i2c(sda,scl)
{
    rawDataBuffer[0] = 0;
    rawDataBuffer[1] = 0;
    status = -1;
    limit = 65535;
}

void BH1750::init(void)
{
    configure(BH1750_POWER_ON);
}

uint16_t BH1750::singleMeas(char mode)
{
    //char data[1];
    uint16_t val = 0;;
    switch (mode) 
    {
        case BH1750_ONE_TIME_HIGH_RES_MODE:
        case BH1750_ONE_TIME_HIGH_RES_MODE_2:
        case BH1750_ONE_TIME_LOW_RES_MODE:
            configure(mode);
            val = readIntesity();
            return(val);
        default:
            return -1;
    }
}

void BH1750::contMeas(char mode)
{
    switch (mode) 
    {
        case BH1750_CONTINUOUS_HIGH_RES_MODE:
        case BH1750_CONTINUOUS_HIGH_RES_MODE_2:
        case BH1750_CONTINUOUS_LOW_RES_MODE:
            configure(mode);
            break;
        default:
            break;
    }
}

void BH1750::configure(char mode)
{
    char data[1];
    switch (mode) 
    {
        case BH1750_CONTINUOUS_HIGH_RES_MODE:
        case BH1750_CONTINUOUS_HIGH_RES_MODE_2:
        case BH1750_CONTINUOUS_LOW_RES_MODE:
        case BH1750_ONE_TIME_HIGH_RES_MODE:
        case BH1750_ONE_TIME_HIGH_RES_MODE_2:
        case BH1750_ONE_TIME_LOW_RES_MODE:
            data[0] = mode;
            status = i2c.write(BH1750_I2CADDR, data, sizeof(data), false);
            wait_ms(10);
            break;
        default:
            break;
    }
}


uint16_t BH1750::readIntesity(void)
{
    status = i2c.read(BH1750_I2CADDR, rawDataBuffer, 2, false);
    if(status != 0) 
    {
        return -1;
    }
    //success
    uint16_t result = 0;
    result = ((rawDataBuffer[0]<<8)|rawDataBuffer[1])/1.2;
    return result;
}

int BH1750::getStatus(void)
{
    return status;
}

char* BH1750::getRawData(void)
{
    return rawDataBuffer;
}

uint16_t BH1750::getLimit(void)
{
    return limit;
}

void BH1750::setLimit(uint16_t lim)
{
    limit = lim;
}

void BH1750::setCurrentLimit(void)
{
    limit = readIntesity();
}