/* mbed library for reading ID3v1 tags in mp3 files
 * Copyright (c) 2010 Christian Schmiljun
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef _MP3_TAG_H
#define _MP3_TAG_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "defines.h"
#include "ff.h"

#define MP3_ID3_TITLE_LENGTH        30
#define MP3_ID3_ARTIST_LENGTH       30
#define MP3_ID3_ALBUM_LENGTH        30
#define MP3_ID3_YEAR_LENGTH          4
#define MP3_ID3_COMMENT_LENGTH      30

/** Struct with the layout of id3tags version 1.
 *
 * @attention Strings are either space- or zero-padded. Unset string entries are filled using an empty string. Strings not allways null-terminated.
 *
 * Size: 128 bytes
 */
typedef struct Id3V1Tag
{
    char tag[3];                                    /*!< "TAG" */
    char title[MP3_ID3_TITLE_LENGTH];               /*!< title, MP3_ID3_TITLE_LENGTH characters */
    char artist[MP3_ID3_ARTIST_LENGTH];             /*!< artist, MP3_ID3_ARTIST_LENGTH characters */
    char album[MP3_ID3_ALBUM_LENGTH];               /*!< album, MP3_ID3_ALBUM_LENGTH characters */
    char year[MP3_ID3_YEAR_LENGTH];                 /*!< year, MP3_ID3_YEAR_LENGTH characters */
    char comment[MP3_ID3_COMMENT_LENGTH];           /*!< comment, MP3_ID3_COMMENT_LENGTH characters or (MP3_ID3_COMMENT_LENGTH - 2), if the track number is stored in the last two bytes */
    char genre;                                     /*!< index in a list of genres, or 255 - see mp3tag_genres */
} Id3V1Tag;

#define MP3_ID3_GENRES 148

#ifdef __cplusplus
extern "C" {
#endif

extern const char* const mp3tag_genres[MP3_ID3_GENRES];  /*!< list of genres, see http://www.multimediasoft.com/amp3dj/help/index.html?amp3dj_00003e.htm*/


/** Read id3tag from a file.
 * 
 * @param openFile
 *   Open/readable audio file. (mp3)
 *
 * @param tag
 *   Return value with id3tag.
 *  
 * @return TRUE on success, tag has the informations; FALSE on failure, tag isn't modiefied, openFile isn't open/readable or file has no id3v1. 
 */
bool mp3tag_readTag(FILE* openFile, Id3V1Tag* tag);

/** Read id3tag from a file.
 * 
 * @param openFile
 *   Open/readable audio file. (mp3)
 *
 * @param tag
 *   Return value with id3tag.
 *  
 * @return TRUE on success, tag has the informations; FALSE on failure, tag isn't modiefied, openFile isn't open/readable or file has no id3v1. 
 */
bool mp3tag_readTagFF(FIL* openFile, Id3V1Tag* tag);

/** Get track number from an id3tag.
 * 
 * @param tag
 *  Id3tag.
 *  
 * @return Track number if exist, or 0. 
 */
char mp3tag_getTrackNumber(Id3V1Tag* tag);

#ifdef __cplusplus
}
#endif

#endif
