/* mbed library for reading ID3v1 tags in mp3 files
 * Copyright (c) 2010 Christian Schmiljun
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mp3tag.h"

// see http://www.multimediasoft.com/amp3dj/help/index.html?amp3dj_00003e.htm
const char* const mp3tag_genres[MP3_ID3_GENRES] = {
  "Blues", "Classic Rock", "Country", "Dance",
  "Disco", "Funk", "Grunge", "Hip-Hop",
  "Jazz", "Metal", "New Age", "Oldies",
  "Other", "Pop", "R&B", "Rap",
  "Reggae", "Rock", "Techno", "Industrial",
  "Alternative", "Ska", "Death Metal", "Pranks",
  "Soundtrack", "Euro-Techno", "Ambient", "Trip-Hop",
  "Vocal", "Jazz+Funk", "Fusion", "Trance",
  "Classical", "Instrumental", "Acid", "House",
  "Game", "Sound Clip", "Gospel", "Noise",
  "AlternRock", "Bass", "Soul", "Punk",
  "Space", "Meditative", "Instrumental Pop", "Instrumental Rock",
  "Ethnic", "Gothic", "Darkwave", "Techno-Industrial",
  "Electronic", "Pop-Folk", "Eurodance", "Dream",
  "Southern Rock", "Comedy", "Cult", "Gangsta",
  "Top 40", "Christian Rap", "Pop/Funk", "Jungle",
  "Native American", "Cabaret", "New Wave", "Psychadelic",
  "Rave", "Showtunes", "Trailer", "Lo-Fi",
  "Tribal", "Acid Punk", "Acid Jazz", "Polka",
  "Retro", "Musical", "Rock & Roll", "Hard Rock",
  "Folk", "Folk/Rock", "National folk", "Swing",
  "Fast-fusion", "Bebob", "Latin", "Revival",
  "Celtic", "Bluegrass", "Avantgarde", "Gothic Rock",
  "Progressive Rock", "Psychedelic Rock", "Symphonic Rock", "Slow Rock",
  "Big Band", "Chorus", "Easy Listening", "Acoustic",
  "Humour", "Speech", "Chanson", "Opera",
  "Chamber Music", "Sonata", "Symphony", "Booty Bass",
  "Primus", "Porn Groove", "Satire", "Slow Jam",
  "Club", "Tango", "Samba", "Folklore",
  "Ballad", "Powder Ballad", "Rhythmic Soul", "Freestyle",
  "Duet", "Punk Rock", "Drum Solo", "A Capella",
  "Euro-House", "Dance Hall", "Goa", "Drum & Bass",
  "Club House", "Hardcore", "Terror", "Indie",
  "BritPop", "NegerPunk", "Polsk Punk", "Beat",
  "Christian Gangsta", "Heavy Metal", "Black Metal", "Crossover",
  "Contemporary C", "Christian Rock", "Merengue", "Salsa",
  "Thrash Metal", "Anime", "JPop", "SynthPop"
};

bool mp3tag_readTag(FILE* openFile, Id3V1Tag* tag)
{
    if (!openFile) 
    {
        DEBUGOUT("MP3Tag: File not open.\r\n");
        return false;
    }
    // backup stream position
    fpos_t pos;
    fgetpos (openFile, &pos);
    if (fseek(openFile, -128, SEEK_END) || fread(tag, 1, 128, openFile) != 128) 
    {
        DEBUGOUT("MP3Tag: Couldn't jump/read in file.\r\n");
        // restore stream position
        fsetpos (openFile, &pos);
        return false;
    }
    // restore stream position
    fsetpos (openFile, &pos);
    if (strncmp(tag->tag, "TAG", 3)) 
    {
        DEBUGOUT("MP3Tag: File doesn't have an ID3 tag.\r\n");
        return false;
    }
    DEBUGOUT("MP3Tag: GenreID %i.\r\n", tag->genre); 
    DEBUGOUT("MP3Tag: Tags readed.\r\n");
    return true;
}

bool mp3tag_readTagFF(FIL* openFile, Id3V1Tag* tag)
{
    if (!openFile) 
    {
        DEBUGOUT("MP3Tag: File not open.\r\n");
        return false;
    }
    // backup stream position
    DWORD pos = openFile->fptr;        
    unsigned int n;
    if (   (f_lseek(openFile, openFile->fsize - 128) != FR_OK) 
        || (f_read(openFile, tag, 128, &n) != FR_OK)
        || n != 128)
    {
        DEBUGOUT("MP3Tag: Couldn't jump/read in file.\r\n");
        // restore stream position
        f_lseek(openFile, pos);
        return false;
    }        
    
    // restore stream position
    f_lseek(openFile, pos);
    if (strncmp(tag->tag, "TAG", 3)) 
    {
        DEBUGOUT("MP3Tag: File doesn't have an ID3 tag.\r\n");
        return false;
    }
    DEBUGOUT("MP3Tag: GenreID %i.\r\n", tag->genre); 
    DEBUGOUT("MP3Tag: Tags readed.\r\n");
    return true;
}

char mp3tag_getTrackNumber(Id3V1Tag* tag)
{
    if (tag->comment[28] == 0x00 &&  tag->comment[29] != 0x00)
    {
        DEBUGOUT("MP3Tag: Tag has track number: %i.\r\n", tag->comment[29]);
        return tag->comment[29];
    }
    DEBUGOUT("MP3Tag: Tag has no track number.\r\n");
    return 0x00;
}

