/* GenericMouse.h */
/* Copyright (c) 2011 ARM Limited. All rights reserved. */

#ifndef _GENERIC_MOUSE_
#define _GENERIC_MOUSE_

#include "USBHID.h"

#define REPORT_ID_MOUSE   2

/* Common usage */
#define MOUSE_LEFT     (1U << 0)   /*!< Left Button */
#define MOUSE_RIGHT    (1U << 1)   /*!< Right Button */
#define MOUSE_MIDDLE   (1U << 2)   /*!< Middle Button */

/* X and Y limits */
/* These values do not directly map to screen pixels */
/* Zero may be interpreted as meaning 'no movement' */
#define X_MIN_ABS    (1)        /*!< Minimum value on x-axis */  
#define Y_MIN_ABS    (1)        /*!< Minimum value on y-axis */
#define X_MAX_ABS    (0x7fff)   /*!< Maximum value on x-axis */
#define Y_MAX_ABS    (0x7fff)   /*!< Maximum value on y-axis */

#define X_MIN_REL    (-127)     /*!< The maximum value that we can move to the left on the x-axis */
#define Y_MIN_REL    (-127)     /*!< The maximum value that we can move up on the y-axis */
#define X_MAX_REL    (127)      /*!< The maximum value that we can move to the right on the x-axis */
#define Y_MAX_REL    (127)      /*!< The maximum value that we can move down on the y-axis */



/** Generic Mouse
 *
 * This class is just an API to use in a child class. See USBMouse.h for instance for more information.
 *
 */
class GenericMouse
{
    public:
        /**
        *   Constructor for a Generic Mouse
        */
        GenericMouse(){ button = 0;};
        
        /**
        * Update the state of the mouse
        *
        * @param x x-axis position (can be absolute or relative)
        * @param y y-axis position (can be absolute or relative)
        * @param buttons buttons state (first bit represents MOUSE_LEFT, second bit MOUSE_RIGHT and third bit MOUSE_MIDDLE)
        * @param z wheel state (>0 to scroll down, <0 to scroll up)
        * @return true if there is no error, false otherwise
        */
        virtual bool update(int16_t x, int16_t y, uint8_t buttons, int8_t z){return false;};
        
        /**
        * Move the cursor to (x, y)
        *
        * @param x x-axis position (can be absolute or relative)
        * @param y y-axis position (can be absolute or relative)
        * @return true if there is no error, false otherwise
        */
        bool move(int16_t x, int16_t y);
        
        /**
        * Press one or several buttons
        *
        * @param button button state (ex: press(MOUSE_LEFT))
        * @return true if there is no error, false otherwise
        */
        bool press(uint8_t button);
        
        /**
        * Release one or several buttons
        *
        * @param button button state (ex: release(MOUSE_LEFT))
        * @return true if there is no error, false otherwise
        */
        bool release(uint8_t button);
        
        /**
        * Double click (MOUSE_LEFT)
        *
        * @return true if there is no error, false otherwise
        */
        bool doubleClick();
        
        /**
        * Click
        *
        * @param button state of the buttons ( ex: clic(MOUSE_LEFT))
        * @return true if there is no error, false otherwise
        */
        bool click(uint8_t button); 
        
        /**
        * Scrolling
        *
        * @param z value of the wheel (>0 to go down, <0 to go up)
        * @return true if there is no error, false otherwise
        */
        bool scroll(int8_t z);
        
    private:
        uint8_t button;
};

#endif
