/* GenericKeyboard.h */
/* Copyright (c) 2011 ARM Limited. All rights reserved. */

#ifndef _GENERIC_KEYBOARD_
#define _GENERIC_KEYBOARD_

#include "USBHID.h"
#include "Stream.h"

#define REPORT_ID_KEYBOARD 1
#define REPORT_ID_VOLUME   3


///All media keys
enum MEDIA_KEY
{
    KEY_NEXT_TRACK,     ///< next Track Button
    KEY_PREVIOUS_TRACK, ///< Previous track Button
    KEY_STOP,           ///< Stop Button
    KEY_PLAY_PAUSE,     ///< Play/Pause Button
    KEY_MUTE,           ///< Mute Button
    KEY_VOLUME_UP,      ///< Volume Up Button
    KEY_VOLUME_DOWN,    ///< Volume Down Button
};


///Different keys
enum KEY
{
    KEY_F1 = 128,   ///< F1 key
    KEY_F2,         ///< F2 key
    KEY_F3,         ///< F3 key
    KEY_F4,         ///< F4 key
    KEY_F5,         ///< F5 key
    KEY_F6,         ///< F6 key
    KEY_F7,         ///< F7 key
    KEY_F8,         ///< F8 key
    KEY_F9,         ///< F9 key
    KEY_F10,        ///< F10 key
    KEY_F11,        ///< F11 key
    KEY_F12,        ///< F12 key
    PRINT_SCREEN,   ///< Print Screen key
    INSERT,         ///< Insert key
    HOME,           ///< Home key
    PAGE_UP,        ///< Page Up key
    PAGE_DOWN,      ///< Page Down key
};


/** Generic keyboard
 *
 * This class is just an API to use in a child class. See USBKeyboard.h for instance for more information.
 *
 */
class GenericKeyboard: public Stream
{
    public:
        /**
        * Constructor for a Generic keyboard
        */
        GenericKeyboard(){};
        
        /**
        * Send all kinds of characters (ctrl + key, ...). 
        *
        * @code
        * //To send CTRL + s (save)
        *  keyboard.keyCode(CTRL, 's');
        * @endcode
        *
        * @param modifier bit 0: CTRL, bit 1: SHIFT, bit 2: ALT
        * @param key character to send
        * @return true if there is no error, false otherwise
        */
        bool keyCode(uint8_t modifier, uint8_t key);
        
        /**
        * Send a character
        *
        * @param c character to be sent
        * @return true if there is no error, false otherwise
        */
        virtual int _putc(int c);
        
        /**
        * Control media keys
        *
        * @param key media key pressed (KEY_NEXT_TRACK, KEY_PREVIOUS_TRACK, KEY_STOP, KEY_PLAY_PAUSE, KEY_MUTE, KEY_VOLUME_UP, KEY_VOLUME_DOWN)
        * @return true if there is no error, false otherwise
        */
        bool mediaControl(MEDIA_KEY key);
        
   private:
        //dummy otherwise it doesn,t compile (we must define all methods of an abstract class)
        virtual int _getc() { return -1;}
        
        
};

#endif
