#ifndef _MAXON_ESCON_H_
#define _MAXON_ESCON_H_

#include "Hallsensor.h"
#include "defines.h"

/**
 * @author Christian Burri
 *
 * @copyright Copyright (c) 2013 HSLU Pren Team #1 Cruising Crêpe
 * All rights reserved.
 *
 * @brief
 *
 * This class implements the driver for the Maxon ESCON servo driver.
 * For more information see the Datasheet:
 * <a href="http://escon.maxonmotor.com">http://escon.maxonmotor.com</a>
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
class MaxonESCON
{

private:

    /** @brief To Enable the amplifier */
    DigitalOut _enb;
    /** @brief Duty Cycle to set the speed */
    PwmOut _pwm;
    /** @brief Hallsensor Class */
    Hallsensor* _hall;
    /** @brief Ready output from ESCON */
    DigitalIn _isenb;
    /** @brief Actual speed from ESCON analog Output 1 */
    AnalogIn _actualSpeed;
    /** @brief increment the Hallpattern */
    int    _pulses;

public:

    /**
     * @brief Create a motor control object.
     * @param enb DigitalOut, set high for enable
     * @param isenb DigitalIn, high for enable
     * @param pwm PwmOut pin, set the velocity
     * @param actualSpeed AnalogIn filtered signal for ActualSpeed from Motor
     * @param hall The object of the hallsensor from Motor
     */
    MaxonESCON(PinName enb,
               PinName isenb,
               PinName pwm,
               PinName actualSpeed,
               Hallsensor *hall);

    /** 
     * @brief Set the speed of the motor with a pwm for 10%..90%.
     * 50% PWM is 0rpm.
     * Caclulate from [1/s] in [1/min] and the factor of the ESCON.
     * @param speed The speed of the motor as a normalised value, given in [1/s]
     */
    void setVelocity(float speed);

    /** 
     * @brief Return the speed from ESCON.
     * 0 rpm is defined in the analog input as 1.65V
     * @return speed of the motor, given in [1/s]
     */
    float getActualSpeed(void);

    /** 
     * @brief Set the period of the pwm duty cycle.
     * Wrapper for PwmOut::period()
     * @param period Pwm duty cycle, given in [s].
     */
    void period(float period);

    /** 
     * @brief Set the motor to a enable sate.
     * @param enb <code>false</code> for disable <code>true</code> for enable.
     */
    void enable(bool enb);

    /**
     * @brief Tests if the servo drive is enabled.
     * @return <code>true</code> if the drive is enabled,
     * <code>false</code> otherwise.
     */
    bool isEnabled(void);

    /**
     * @brief Return the number of pulses.
     * @return Pulses, given in [count]
     */
    int getPulses(void);

    /**
     * @brief Set the pulses of the motor, given in [count]
     * @return Pulses, given in [count]
     */
    int setPulses(int setPos);
};

#endif
