#ifndef HALLSENSOR_H
#define HALLSENSOR_H

#include "mbed.h"

/**
 * @author Christian Burri
 *
 * @copyright Copyright (c) 2013 HSLU Pren Team #1 Cruising Crêpe
 * All rights reserved.
 *
 * @brief
 *
 * Interface to count the Hallsensor input from an EC-Motor.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
class Hallsensor
{

private:

    /**
     * @brief Update the pulse count.
     * Called on every rising/falling edge of Hall 1-3.
     * Reads the state of the channels and determines whether a pulse forward
     * or backward has occured, updating the count appropriately.
     */
    void encode(void);

    InterruptIn hall1_;
    InterruptIn hall2_;
    InterruptIn hall3_;

    int          prevState_;
    int          currState_;

    volatile int pulses_;

public:

    /**
     * @brief Constructor of the class <code>Hallsensor</code>.
     *
     * Reads the current values on Hall1, Hall2 and Hall3 to determine the
     * initial state.
     * Attaches the encode function to the rise/fall interrupt edges of
     * Hall1, Hall2 and Hall3.
     * @param hall1    mbed pin for Hall1 input.
     * @param hall2    mbed pin for Hall2 input.
     * @param hall3    mbed pin for Hall3 input.
     */
    Hallsensor(PinName hall1, PinName hall2, PinName hall3);

    /**
     * @brief Reset the encoder.
     * Sets the pulses and revolutions count to zero.
     */
    void reset(void);

    /**
     * @brief Read the number of pulses recorded by the encoder.
     * @return Number of pulses which have occured, given in [count]
     */
    int getPulses(void);

    /**
     * @brief Read the number of revolutions recorded by the encoder.
     * @return Number of revolutions which have occured on the index channel.
     */
    int getRevolutions(void);

};

#endif
