#ifndef _DEFINES_H_
#define _DEFINES_H_

/*! \file defines.h
\brief All defines for the roboter you can see here.
*/
#include "mbed.h"

/**
 * @name Physical dimensions &pi;;
 * @{
 */
#define PI                    3.141592654f
/*! @} */

/**
 * @name maxon motor #339282 EC 45 flat 30W
 * @{
 */

/**
* @brief Number of of pole pairs
*/
#define POLE_PAIRS            8u

/**
* @brief Gear on the motor 1/11.6f
*/
#define GEAR                  1/11.6f

/**
* @brief Pulses per electrical step form the Hallsensor, have 6 steps
*/
#define PULSES_PER_STEP       6u
/*! @} */

/**
 * @name Physical Dimension of the car
 * @{
 */

/**
 * @brief Value for the diffrerenz between left an right, given in [m]
 */
#define WHEEL_RADIUS_DIFF     0.0001f

/**
 * @brief Radius of the left wheel, given in [m] kleiner --> weiter
 */
#define WHEEL_RADIUS_LEFT     0.040190f            

/**
 * @brief Radius of the left wheel, given in [m]
 */
#define WHEEL_RADIUS_RIGHT    (WHEEL_RADIUS_LEFT - WHEEL_RADIUS_DIFF)

/**
 * @brief Distance of the wheel, given in [m] Greater --> turn more
 */
#define WHEEL_DISTANCE        (0.1700f-0.001f)  // org.0.17500f alt: 0.173f  //(0.1700f-0.004f) immer zu weit links  (0.1700f-0.006f) war mal gut aber jetz zu weit rechts 

/**
 * @brief Sets the start X-point, given in [m]
 */
#define START_X_OFFSET        -0.850f//-0.8f

/**
 * @brief Sets the start Y-point, given in [m]
 */
#define START_Y_OFFSET        0.775f//0.8f
/*! @} */

/**
 * @name State Bits of the car
 * @{
 */

/**
 * @brief Bit0 = stop pressed
 */
#define STATE_STOP            1u

/**
 * @brief Bit1 = Undervoltage battery
 */
#define STATE_UNDER           2u

/**
 * @brief Bit2 = left ESCON in error state
 */
#define STATE_LEFT            4u

/**
 * @brief Bit3 = right ESCON in error state
 */
#define STATE_RIGHT           8u
/*! @} */

/**
 * @name ESCON Constands
 *
 * @{
 */

/**
 * @brief Speed Factor how set in the ESCON Studio
 */
#define ESCON_SET_FACTOR      2400.0f

/**
 * @brief Speed Factor how get in the ESCON Studio
 */
#define ESCON_GET_FACTOR      2500.0f

/**
 * @brief Error patch of the drift of Analog input and pwn output for set speed
 */
#define SET_SPEED_PATCH       (1+0.00262f)

/**
 * @brief Error patch of the drift of Analog input and pwn output for get speed
 */
#define GET_SPEED_PATCH       (1+0.0019f)
/*! @} */

/**
 * @name position controller
 * @{
 */

/**
 * @brief Main Gain for k1, k2 and k3
 */
#define GAIN                  0.65f

/**
 * @brief Gain k1 default 1.0f
 */
#define K1                    1.45f * GAIN

/**
 * @brief Gain k2 default 3.0f
 */
#define K2                    3.0f * GAIN

/**
 * @brief Gain k3 default 2.0f
 */
#define K3                    2.0f * GAIN

/**
 * @brief Min. Distance to switch the position controller off.
 * Because when Distance Error goes to zero the ATAN2 is not define, given in [m]
 */
#define MIN_DISTANCE_ERROR    0.045f
/*! @} */

/**
 * @name Batterie control Battery Type: 1SP1P LG-18650
 * nominal voltage 3.6V
 * 5 batterys ==> 5 * 3.5V = 17.5V
 * @{
 */

/**
 * @brief Battery Multiplicator for the potential divider.
 * 
 * R2 / (R1 + R2) = 0.153    R2= 10k , R1 = 1.8k 1/0.153 = 6.555  --> 3.3 * 6.555 = 21.6333333f
 */
#define BAT_MULTIPLICATOR     21.633333333f

/**
 * @brief minium operate voltage, given in [V]
 */
#define BAT_MIN               17.5f
/*! @} */

/**
 * @name sampling rate for a Task Object
 * @{
 */

/**
 * @brief 1kHz Rate for Robot Control, given in [s]
 */
#define PERIOD_ROBOTCONTROL   0.001f

/**
 * @brief 1kHz Rate for State Objekt , given in [s]
 */
#define PERIOD_STATE          0.001f

/**
 * @brief 10Hz Rate for the Android communication , given in [s]
 */
#define PERIOD_ANDROID        (0.035f*2) //(0.025f*2)
/*! @} */


/**
 * @brief struct state
 * structure containing system sensor data
 */
typedef struct state {
    /** @brief millis Time [ms]*/
    int millis;
    /** @brief Battery voltage [V] */
    float voltageBattery;
    /** @brief Number of pulses left */
    int leftPulses;
    /** @brief Number of pulses right */
    int rightPulses;
    /** @brief Velocity left [m/s] */
    float leftVelocity;
    /** @brief Velocity right [m/s] */
    float rightVelocity;
    /** @brief Velocity of the car [m/s] */
    float velocity;
    /** @brief Velocity rotation [°/s] */
    float omega;
    /** @brief X-Axis from co-ordinate [m] */
    float xAxis;
    /** @brief Y-Axis from co-ordinate [m] */
    float yAxis;
    /** @brief X-Axis Error [m] */
    float xAxisError;
    /** @brief X-Axis Error [m] */
    float yAxisError;
    /** @brief Angle Error [°] */
    float angleError;
    /** @brief Angle from Car [°] */
    float angle;
    /** @brief Setpoint X-Axis [m] */
    float setxAxis;
    /** @brief Setpoint Y-Axis [m] */
    float setyAxis;
    /** @brief Setpoint Angel [°] */
    float setAngle;
    /** @brief Setpoint velocitiy [m/s] */
    float setVelocity;
    /** @brief Setpoint rotation velocitiy [rad/s] */
    float setOmega;
    /** @brief State of the car */
    int state;
    /** @brief distance to Goal */
    float rho;
    /** @brief theta to goal */
    float lamda;
    /** @brief theta from the goal */
    float delta;
} state_t;



#endif
